package ru.yandex.solomon.expression.expr.func.util;

import java.time.Instant;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValueInterval;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListOrView;
import ru.yandex.solomon.util.time.Interval;

/**
 * <p>Returns interval of first and last points for specified timeseries. Interval boarders extends if
 * specified more then one timeseries.
 * <p>Example usage {@code
 * let interval = time_interval(graphData);
 * let cropped = crop(anotherGraphData, interval);
 * }
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelFnTimeInterval implements SelFuncProvider {

    private static Interval interval(NamedGraphData graphData) {
        AggrGraphDataArrayListOrView target = graphData.getAggrGraphDataArrayList();
        if (target.isEmpty()) {
            return new Interval(Instant.EPOCH, Instant.EPOCH);
        }

        long fromMillis = target.getTsMillis(0);
        long toMillis = target.getTsMillis(target.length() - 1);
        return Interval.millis(fromMillis, toMillis);
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name("time_interval")
            .help("Returns interval of first and last points for specified timeseries. Interval boarders extends if " +
                "specified more then one timeseries")
            .category(SelFuncCategory.OTHER)
            .args(SelTypes.GRAPH_DATA)
            .returnType(SelTypes.INTERVAL)
            .handler(args -> {
                var source = args.get(0).castToGraphData().getNamedGraphData();
                return new SelValueInterval(interval(source));
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("time_interval")
            .help("Returns interval of first and last points for specified timeseries. Interval boarders extends if " +
                "specified more then one timeseries")
            .category(SelFuncCategory.OTHER)
            .args(SelTypes.GRAPH_DATA_VECTOR)
            .returnType(SelTypes.INTERVAL)
            .handler(args -> {
                Interval result = Stream.of(args.get(0).castToVector().valueArray())
                    .map(selValue -> interval(selValue.castToGraphData().getNamedGraphData()))
                    .filter(interval -> !Interval.EMPTY.equals(interval))
                    .reduce(Interval::convexHull)
                    .orElse(Interval.EMPTY);

                return new SelValueInterval(result);
            })
            .build());
    }
}
