package ru.yandex.solomon.expression.expr.op.bin;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BinaryOperator;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * Utilities to combine vectors of metrics by labels "one-to-one".
 * This method finds a unique pair of entries from each source.
 * Two entries match if they have exact same labels except different common labels in each source.
 */
class OneToOneMatcher {
    @FunctionalInterface
    public interface BiGraphDataFunction extends BinaryOperator<GraphData> {
    }

    public static SelValue[] match(
            NamedGraphData[] aVector,
            NamedGraphData[] bVector,
            BiGraphDataFunction fn)
    {
        Pair<NamedGraphData[], NamedGraphData[]> pair =
            dropCommonMetricNameAndLabels(aVector, bVector);

        return oneToOneMatchingImpl(pair.getLeft(), pair.getRight(), fn);
    }

    private static Pair<NamedGraphData[], NamedGraphData[]> dropCommonMetricNameAndLabels(
        NamedGraphData[] aVector,
        NamedGraphData[] bVector)
    {
        /*
         * Drop metric name in case:
         * - errors{...}
         * - requests{...}
         * In other case metric name drop has same behavior with common labels drops
         */
        String aCommonMetricName = getCommonMetricNameOrNull(aVector);
        String bCommonMetricName = getCommonMetricNameOrNull(bVector);

        boolean dropMetricName = aCommonMetricName != null
            && bCommonMetricName != null
            && !aCommonMetricName.equals(bCommonMetricName);

        /*
         * We drop common label keys for each source that have different values in each source:
         *
         * Example 1 (different "code"):
         * let source1 = errors{code=500, endpoint=*};
         * let source2 = errors{code=5xx, endpoint=*};
         * return source1 / source2;
         *
         * Example 2 (additional labels in one of sources):
         * let source1 = errors{code=500, endpoint=*};
         * let source2 = totalErrors{endpoint=*};
         * return source1 / source2;
         *
         * Please note case of common label in one source and not common label in other source:
         * errors{code=500, endpoint=...}
         *
         * requests{code=400, endpoint=...}
         * requests{code=500, endpoint=...}
         *
         * We don't drop code label in that case!
         */
        Set<String> allKeys = getAllKeys(aVector, bVector);

        Map<String, String> aCommonLabelsMap = new HashMap<>(allKeys.size());
        Set<String> aNotCommonLabelKeys = new HashSet<>(allKeys.size());
        fillCommonAndNotCommonLabels(aVector, aCommonLabelsMap, aNotCommonLabelKeys);

        Map<String, String> bCommonLabelsMap = new HashMap<>(allKeys.size());
        Set<String> bNotCommonLabelKeys = new HashSet<>(allKeys.size());
        fillCommonAndNotCommonLabels(bVector, bCommonLabelsMap, bNotCommonLabelKeys);

        Set<String> aDropLabelKeys = new HashSet<>();
        Set<String> bDropLabelKeys = new HashSet<>();

        for (String key : allKeys) {
            String aValue = aCommonLabelsMap.get(key);
            String bValue = bCommonLabelsMap.get(key);

            if (aValue == null && bValue == null) {
                continue;
            }

            if (aValue != null && bNotCommonLabelKeys.contains(key)
                || bValue != null && aNotCommonLabelKeys.contains(key))
            {
                continue;
            }

            if (!Objects.equals(aValue, bValue)) {
                aDropLabelKeys.add(key);
                bDropLabelKeys.add(key);
            }
        }

        NamedGraphData[] aPreparedVector =
            dropCommonLabelsAndMetricNames(aVector, aDropLabelKeys, dropMetricName);
        NamedGraphData[] bPreparedVector =
            dropCommonLabelsAndMetricNames(bVector, bDropLabelKeys, dropMetricName);

        return Pair.of(aPreparedVector, bPreparedVector);
    }

    private static Set<String> getAllKeys(NamedGraphData[] aVector, NamedGraphData[] bVector) {
        return Stream.concat(Arrays.stream(aVector), Arrays.stream(bVector))
            .map(NamedGraphData::getLabels)
            .flatMap(Labels::stream)
            .map(Label::getKey)
            .collect(Collectors.toSet());
    }

    private static void fillCommonAndNotCommonLabels(
        NamedGraphData[] aVector,
        Map<String, String> commonLabelsMap,
        Set<String> notCommonLabels)
    {
        Arrays.stream(aVector)
            .map(NamedGraphData::getLabels)
            .flatMap(Labels::stream)
            .forEach(label -> {
                if (!notCommonLabels.contains(label.getKey())) {
                    String commonValue = commonLabelsMap.get(label.getKey());

                    if (commonValue == null) {
                        commonLabelsMap.put(label.getKey(), label.getValue());
                    } else if (!commonValue.equals(label.getValue())) {
                        commonLabelsMap.remove(label.getKey());
                        notCommonLabels.add(label.getKey());
                    }
                }
            });
    }

    private static String getCommonMetricNameOrNull(NamedGraphData[] vector) {
        Set<String> metricNames = Arrays.stream(vector)
            .map(NamedGraphData::getMetricName)
            .collect(Collectors.toSet());

        if (metricNames.size() == 1) {
            return metricNames.iterator().next();
        }

        return null;
    }

    private static SelValue[] oneToOneMatchingImpl(
            NamedGraphData[] aVector,
            NamedGraphData[] bVector,
            BiGraphDataFunction fn)
    {
        List<SelValue> result = new ArrayList<>(aVector.length);

        for (NamedGraphData a : aVector) {
            for (NamedGraphData b : bVector) {
                if (a.getLabels().equals(b.getLabels()) && a.getMetricName().equals(b.getMetricName())) {
                    GraphData graphData = fn.apply(a.getGraphData(), b.getGraphData());
                    NamedGraphData prepared = NamedGraphData.newBuilder()
                        .setGraphData(graphData)
                        .setMetricName(a.getMetricName())
                        .setLabels(a.getLabels())
                        .build();
                    result.add(new SelValueGraphData(prepared));
                }
            }
        }

        return result.toArray(new SelValue[0]);
    }

    private static NamedGraphData[] dropCommonLabelsAndMetricNames(
        NamedGraphData[] vector,
        Set<String> commonLabelKeys,
        boolean dropMetricName)
    {
        return Arrays.stream(vector)
            .map(ngd -> dropCommonLabelsAndMetricNames(ngd, commonLabelKeys, dropMetricName))
            .toArray(NamedGraphData[]::new);
    }

    private static NamedGraphData dropCommonLabelsAndMetricNames(
        NamedGraphData ngd,
        Set<String> commonLabelKeys,
        boolean dropMetricName)
    {
        String newMetricName = dropMetricName ? "" : Nullables.orEmpty(ngd.getMetricName());

        Labels newLabels = removeLabelsByKeys(ngd.getLabels(), commonLabelKeys);

        return new NamedGraphData(
            "",
            ngd.getType(),
            newMetricName,
            newLabels,
            ngd.getDataType(),
            ngd.getAggrGraphDataArrayList()
        );
    }

    private static Labels removeLabelsByKeys(Labels labels, Set<String> keys) {
        if (keys.isEmpty()) {
            return labels;
        }

        LabelsBuilder builder = labels.toBuilder();
        keys.forEach(builder::remove);
        return builder.build();
    }
}
