package ru.yandex.solomon.expression.expr.op.bin;

import java.util.function.DoubleBinaryOperator;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.expr.func.SelOp;
import ru.yandex.solomon.expression.expr.func.SelOpProvider;
import ru.yandex.solomon.expression.expr.func.SelOpRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueDouble;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;

/**
 * @author Vladimir Gordiychuk
 */
public class SelOpArithDouble implements SelOpProvider {
    private static double[] forEachScalar(double[] left, double right, DoubleBinaryOperator op) {
        double[] result = new double[left.length];
        for (int index = 0; index < left.length; index++) {
            result[index] = op.applyAsDouble(left[index], right);
        }
        return result;
    }

    private static NamedGraphData forEachPoint(NamedGraphData left, double right, DoubleBinaryOperator op) {
        var source = left.getAggrGraphDataArrayList();
        if (source.isEmpty()) {
            return left;
        }

        var result = new AggrGraphDataArrayList(StockpileColumns.minColumnSet(MetricType.DGAUGE), source.length());
        var it = MetricTypeTransfers.of(left.getDataType(), MetricType.DGAUGE, source.iterator());
        var point = RecyclableAggrPoint.newInstance();
        var writePoint = RecyclableAggrPoint.newInstance();
        writePoint.reset();
        try {
            while (it.next(point)) {
                writePoint.setTsMillis(point.getTsMillis());
                writePoint.setValue(op.applyAsDouble(point.getValueDivided(), right));
                result.addRecord(writePoint);
            }
            return left.toBuilder()
                .setType(ru.yandex.monlib.metrics.MetricType.DGAUGE)
                .setGraphData(MetricType.DGAUGE, result)
                .build();
        } finally {
            point.recycle();
            writePoint.recycle();
        }
    }

    private static SelValue[] forEachGraph(SelValue[] left, double right, DoubleBinaryOperator op) {
        var result = new SelValue[left.length];
        for (int index = 0; index < left.length; index++) {
            var item = left[index].castToGraphData().getNamedGraphData();
            result[index] = new SelValueGraphData(forEachPoint(item, right, op));
        }
        return result;
    }

    private SelOp opScalarAndScalar(DoubleArithBinOp op) {
        return SelOp.newBuilder()
            .name(op.getName())
            .operator(op.getOperator())
            .args(SelTypes.DOUBLE, SelTypes.DOUBLE)
            .returnType(SelTypes.DOUBLE)
            .handler(args -> {
                var left = args.get(0).castToScalar().getValue();
                var right = args.get(1).castToScalar().getValue();
                return new SelValueDouble(op.apply(left, right));
            })
            .build();
    }

    private SelOp opScalarAndVectorScalar(DoubleArithBinOp op) {
        return SelOp.newBuilder()
            .name(op.getName())
            .operator(op.getOperator())
            .args(SelTypes.DOUBLE, SelTypes.DOUBLE_VECTOR)
            .returnType(SelTypes.DOUBLE_VECTOR)
            .handler(args -> {
                var left = args.get(0).castToScalar().getValue();
                var right = args.get(1).castToVector().doubleArray();
                return new SelValueVector(forEachScalar(right, left, op::reverse));
            })
            .build();
    }

    private SelOp opVectorScalarAndScalar(DoubleArithBinOp op) {
        return SelOp.newBuilder()
            .name(op.getName())
            .operator(op.getOperator())
            .args(SelTypes.DOUBLE_VECTOR, SelTypes.DOUBLE)
            .returnType(SelTypes.DOUBLE_VECTOR)
            .handler(args -> {
                var left = args.get(0).castToVector().doubleArray();
                var right = args.get(1).castToScalar().getValue();
                return new SelValueVector(forEachScalar(left, right, op::apply));
            })
            .build();
    }

    private SelOp opScalarAndGraph(DoubleArithBinOp op) {
        return SelOp.newBuilder()
            .name(op.getName())
            .operator(op.getOperator())
            .args(SelTypes.DOUBLE, SelTypes.GRAPH_DATA)
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var left = args.get(0).castToScalar().getValue();
                var right = args.get(1).castToGraphData().getNamedGraphData();
                return new SelValueGraphData(forEachPoint(right, left, op::reverse));
            })
            .build();
    }

    private SelOp opGraphAndScalar(DoubleArithBinOp op) {
        return SelOp.newBuilder()
            .name(op.getName())
            .operator(op.getOperator())
            .args(SelTypes.GRAPH_DATA, SelTypes.DOUBLE)
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var left = args.get(0).castToGraphData().getNamedGraphData();
                var right = args.get(1).castToScalar().getValue();
                return new SelValueGraphData(forEachPoint(left, right, op::apply));
            })
            .build();
    }

    private SelOp opScalarAndVectorGraph(DoubleArithBinOp op) {
        return SelOp.newBuilder()
            .name(op.getName())
            .operator(op.getOperator())
            .args(SelTypes.DOUBLE, SelTypes.GRAPH_DATA_VECTOR)
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> {
                var left = args.get(0).castToScalar().getValue();
                var right = args.get(1).castToVector().valueArray();
                return new SelValueVector(SelTypes.GRAPH_DATA, forEachGraph(right, left, op::reverse));
            })
            .build();
    }

    private SelOp opVectorGraphAndScalar(DoubleArithBinOp op) {
        return SelOp.newBuilder()
            .name(op.getName())
            .operator(op.getOperator())
            .args(SelTypes.GRAPH_DATA_VECTOR, SelTypes.DOUBLE)
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> {
                var left = args.get(0).castToVector().valueArray();
                var right = args.get(1).castToScalar().getValue();
                return new SelValueVector(SelTypes.GRAPH_DATA, forEachGraph(left, right, op::apply));
            })
            .build();
    }

    @Override
    public void provide(SelOpRegistry registry) {
        for (DoubleArithBinOp op : DoubleArithBinOp.values()) {
            add(registry, op);
        }
    }

    public void add(SelOpRegistry registry, DoubleArithBinOp op) {
        // double + double
        registry.add(opScalarAndScalar(op));
        // double + double[]
        registry.add(opScalarAndVectorScalar(op));
        // double[] + double
        registry.add(opVectorScalarAndScalar(op));
        // double + graph
        registry.add(opScalarAndGraph(op));
        // graph + double
        registry.add(opGraphAndScalar(op));
        // double + graph[]
        registry.add(opScalarAndVectorGraph(op));
        // graph[] + double
        registry.add(opVectorGraphAndScalar(op));
    }
}
