package ru.yandex.solomon.expression.expr.op.bin;

import java.util.List;
import java.util.function.DoubleBinaryOperator;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.expr.func.LabelsUtil;
import ru.yandex.solomon.expression.expr.func.SelOp;
import ru.yandex.solomon.expression.expr.func.SelOpProvider;
import ru.yandex.solomon.expression.expr.func.SelOpRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.math.Resampler;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.Timeline;

/**
 * @author Vladimir Gordiychuk
 */
public class SelOpArithGraph implements SelOpProvider {

    static GraphData graphAndGraph(PositionRange range, GraphData a, GraphData b, DoubleBinaryOperator op) {
        Timeline timeline = Timeline.union(a.getTimeline(), b.getTimeline());

        GraphData ar = Resampler.resample(a, timeline);
        GraphData br = Resampler.resample(b, timeline);

        var result = SelOpArithVectorDouble.vectorDoubleAndVectorDouble(range, ar.getValues().copyOrArray(), br.getValues().copyOrArray(), op);
        return new GraphData(timeline, result);
    }

    private static NamedGraphData graphAndGraph(PositionRange range, NamedGraphData a, NamedGraphData b, DoubleBinaryOperator op) {
        GraphData graphData = graphAndGraph(range, a.getGraphData(), b.getGraphData(), op);

        String metricName = a.getMetricName().equals(b.getMetricName())
            ? a.getMetricName()
            : "";

        Labels commonLabels = LabelsUtil.getCommonLabels(List.of(a, b));

        return NamedGraphData.newBuilder()
            .setGraphData(graphData)
            .setMetricName(metricName)
            .setLabels(commonLabels)
            .build();
    }

    static SelValue graphAndGraph(PositionRange range, SelValue left, SelValue right, DoubleBinaryOperator op) {
        return new SelValueGraphData(graphAndGraph(range, left.castToGraphData().getNamedGraphData(), right.castToGraphData().getNamedGraphData(), op));
    }

    static SelValue[] forEachGraph(boolean singletonVectorLikeScalar, PositionRange range, SelValue[] left, SelValue right, DoubleBinaryOperator op) {
        if (left.length == 1 && singletonVectorLikeScalar) {
            return new SelValue[]{graphAndGraph(range, left[0], right, op)};
        }
        GraphData rightGraphData = right.castToGraphData().getGraphData();
        var result = new SelValue[left.length];
        for (int index = 0; index < left.length; index++) {
            var item = left[index].castToGraphData().getNamedGraphData();
            var itemResult = graphAndGraph(range, item.getGraphData(), rightGraphData, op);
            result[index] = new SelValueGraphData(item.toBuilder()
                .setGraphData(itemResult)
                .build());
        }
        return result;
    }

    private SelOp opGraphAndGraph(DoubleArithBinOp op) {
        return SelOp.newBuilder()
            .name(op.getName())
            .operator(op.getOperator())
            .args(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA)
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var left = args.get(0).castToGraphData().getNamedGraphData();
                var right = args.get(1).castToGraphData().getNamedGraphData();
                PositionRange leftRange = args.getRange(0);
                PositionRange rightRange = args.getRange(1);
                return new SelValueGraphData(graphAndGraph(PositionRange.convexHull(leftRange, rightRange),
                    left, right, op::apply));
            })
            .build();
    }

    private SelOp opGraphAndVectorGraph(DoubleArithBinOp op) {
        return SelOp.newBuilder()
            .name(op.getName())
            .operator(op.getOperator())
            .args(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA_VECTOR)
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler((ctx, args) -> {
                var left = args.get(0);
                var right = args.get(1).castToVector().valueArray();
                PositionRange leftRange = args.getRange(0);
                PositionRange rightRange = args.getRange(1);
                boolean singletonVectorLikeScalar = SelVersion.GROUP_LINES_RETURN_VECTOR_2.since(ctx.getVersion());
                return new SelValueVector(SelTypes.GRAPH_DATA, forEachGraph(singletonVectorLikeScalar,
                        PositionRange.convexHull(leftRange, rightRange), right, left, op::reverse));
            })
            .build();
    }

    private SelOp opVectorGraphAndGraph(DoubleArithBinOp op) {
        return SelOp.newBuilder()
            .name(op.getName())
            .operator(op.getOperator())
            .args(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA)
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler((ctx, args) -> {
                var left = args.get(0).castToVector().valueArray();
                var right = args.get(1);
                PositionRange leftRange = args.getRange(0);
                PositionRange rightRange = args.getRange(1);
                boolean singletonVectorLikeScalar = SelVersion.GROUP_LINES_RETURN_VECTOR_2.since(ctx.getVersion());
                return new SelValueVector(SelTypes.GRAPH_DATA, forEachGraph(singletonVectorLikeScalar,
                        PositionRange.convexHull(leftRange, rightRange), left, right, op::apply));
            })
            .build();
    }

    @Override
    public void provide(SelOpRegistry registry) {
        for (DoubleArithBinOp op : DoubleArithBinOp.values()) {
            add(registry, op);
        }
    }

    public void add(SelOpRegistry registry, DoubleArithBinOp op) {
        // graph + graph
        registry.add(opGraphAndGraph(op));
        // graph + graph[]
        registry.add(opGraphAndVectorGraph(op));
        // graph[] + graph
        registry.add(opVectorGraphAndGraph(op));
    }
}
