package ru.yandex.solomon.expression.parser;

import java.time.Duration;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class DurationIntervalsUnion {
    private static final Pattern INTERVAL_PATTERN = Pattern.compile("^\\[(\\w+)-(\\w+)\\]$");
    private final DurationInterval[] union;

    private static class DurationInterval {
        private final long beginMillis;
        private final long endMillis;

        private DurationInterval(long begin, long end) {
            if (begin > end) {
                throw new IllegalArgumentException("begin is not less or equal than end: " + begin + ", " + end);
            }

            beginMillis = begin;
            endMillis = end;
        }
        boolean contains(long tsMillis) {
            return (beginMillis <= tsMillis) && (tsMillis <= endMillis);
        }
        boolean overlaps(DurationInterval next) {
            return next.beginMillis <= endMillis;
        }
        public static DurationInterval parse(String intervalStr) {
            Matcher matcher = INTERVAL_PATTERN.matcher(intervalStr);
            if (!matcher.find()) {
                throw new IllegalArgumentException("Failed to parse interval (should look like [1h30m-18h]), but was: " + intervalStr);
            }
            Duration begin = DurationParser.parseDuration(matcher.group(1));
            Duration end = DurationParser.parseDuration(matcher.group(2));
            return new DurationInterval(begin.toMillis(), end.toMillis());
        }
        public static int compareByBegin(DurationInterval a, DurationInterval b) {
            return Long.signum(a.beginMillis - b.beginMillis);
        }
        @Override
        public String toString() {
            Duration b = Duration.ofMillis(beginMillis);
            Duration e = Duration.ofMillis(endMillis);
            return "[" + DurationParser.formatDuration(b) + "-" + DurationParser.formatDuration(e) + "]";
        }
    }

    public boolean contains(long tsMillis) {
        return Stream.of(union).anyMatch(di -> di.contains(tsMillis));
    }

    private DurationIntervalsUnion(DurationInterval[] union) {
        this.union = union;
    }

    /**
     * Parses strings like "[1h-1h30m] + [19h30m-23h]", does sanity checks and sorts
     * intervals
     */
    public static DurationIntervalsUnion parse(String code) {
        DurationInterval[] union = Stream.of(code.replaceAll("\\s", "").split("\\+"))
            .map(DurationInterval::parse)
            .toArray(DurationInterval[]::new);

        Arrays.sort(union, DurationInterval::compareByBegin);
        for (int i = 0; i < union.length - 1; i++) {
            if (union[i].overlaps(union[i+1])) {
                throw new IllegalArgumentException("Found overlapping intervals: " + union[i] + " and " + union[i + 1]);
            }
        }

        return new DurationIntervalsUnion(union);
    }

}
