package ru.yandex.solomon.expression.parser;

import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DurationParser {
    private static final long MILLISECONDS_PER_WEEK = TimeUnit.DAYS.toMillis(7);
    private static final long MILLISECONDS_PER_DAY = TimeUnit.DAYS.toMillis(1);
    private static final long MILLISECONDS_PER_HOUR = TimeUnit.HOURS.toMillis(1);
    private static final long MILLISECONDS_PER_MINUTE = TimeUnit.MINUTES.toMillis(1);
    private static final long MILLISECONDS_PER_SECOND = TimeUnit.SECONDS.toMillis(1);

    private static final Pattern DURATION_PATTERN = Pattern.compile("^(\\d+)(ms|[wdhms])");

    @Nonnull
    public static Duration parseDuration(@Nullable String duration) {
        if (duration == null || duration.isEmpty()) {
            throw new IllegalArgumentException("Duration can't be null or empty");
        }

        String unparsedDuration = duration;

        boolean negative = false;
        if (duration.startsWith("-")) {
            negative = true;
            unparsedDuration = duration.substring(1);
        }

        if (!DURATION_PATTERN.asPredicate().test(unparsedDuration)) {
            throw new IllegalArgumentException("Not able to parse duration in format [" + duration + "] example available format [1w2d3h4m5s]");
        }

        long sumInMilliseconds = 0;
        while (unparsedDuration.length() > 0) {
            Matcher matcher = DURATION_PATTERN.matcher(unparsedDuration);
            if (!matcher.find()) {
                break;
            }

            int count = Integer.parseInt(matcher.group(1));

            String unit = matcher.group(2);
            switch (unit) {
                case "w" -> sumInMilliseconds += count * MILLISECONDS_PER_WEEK;
                case "d" -> sumInMilliseconds += count * MILLISECONDS_PER_DAY;
                case "h" -> sumInMilliseconds += count * MILLISECONDS_PER_HOUR;
                case "m" -> sumInMilliseconds += count * MILLISECONDS_PER_MINUTE;
                case "s" -> sumInMilliseconds += count * MILLISECONDS_PER_SECOND;
                case "ms" -> sumInMilliseconds += count;
                default -> throw new UnsupportedOperationException("Unsupported unit [" + unit + "] in duration " + duration);
            }

            unparsedDuration = unparsedDuration.substring(matcher.group(0).length());
        }

        if (negative) {
            sumInMilliseconds = -sumInMilliseconds;
        }

        return  Duration.ofMillis(sumInMilliseconds);
    }

    @Nonnull
    public static String formatDuration(Duration duration) {
        StringBuilder sb = new StringBuilder();
        long milliseconds = duration.toMillis();
        if (duration.isNegative()) {
            sb.append("-");
            milliseconds *= -1;
        }

        long weeks = milliseconds / MILLISECONDS_PER_WEEK;
        milliseconds -= weeks * MILLISECONDS_PER_WEEK;
        if (weeks != 0) {
            sb.append(weeks).append("w");
        }

        long days = milliseconds / MILLISECONDS_PER_DAY;
        milliseconds -= days * MILLISECONDS_PER_DAY;
        if (days != 0) {
            sb.append(days).append("d");
        }

        long hours = milliseconds / MILLISECONDS_PER_HOUR;
        milliseconds -= hours * MILLISECONDS_PER_HOUR;
        if (hours != 0) {
            sb.append(hours).append("h");
        }

        long minutes = milliseconds / MILLISECONDS_PER_MINUTE;
        milliseconds -= minutes * MILLISECONDS_PER_MINUTE;
        if (minutes != 0) {
            sb.append(minutes).append("m");
        }

        long seconds = milliseconds / MILLISECONDS_PER_SECOND;
        milliseconds -= seconds * MILLISECONDS_PER_SECOND;
        if (seconds != 0) {
            sb.append(seconds).append("s");
        }

        if (milliseconds != 0) {
            sb.append(milliseconds).append("ms");
        }

        if (sb.length() < 2) {
            return "0s";
        }

        return sb.toString();
    }

}
