package ru.yandex.solomon.expression.value;

import java.util.Arrays;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypeVector;
import ru.yandex.solomon.expression.type.SelTypes;

/**
 * @author Maksim Leonov
 */
@ParametersAreNonnullByDefault
public class SelValueVector extends SelValue {
    @Nullable
    public final SelValue[] array;
    @Nullable
    public final double[] doubles;
    public final SelType elementType;

    public SelValueVector(SelType elementType, SelValue[] array) {
        if (elementType == SelTypes.DOUBLE) {
            throw new IllegalArgumentException("Using generic SelValueVector for double array");
        }

        for (SelValue item : array) {
            if (!elementType.equals(item.type())) {
                throw new IllegalArgumentException("Not valid type of vector element, expected " + elementType + ", but was " + item.type());
            }
        }

        this.elementType = elementType;
        this.array = array;
        this.doubles = null;
    }

    public SelValueVector(double[] doubles) {
        this.elementType = SelTypes.DOUBLE;
        this.array = null;
        this.doubles = doubles;
    }

    public int length() {
        return match(new Match<Integer>() {
            @Override
            public Integer regular(SelValue[] array, SelType elementType) {
                return array.length;
            }

            @Override
            public Integer doubles(double[] doubles) {
                return doubles.length;
            }
        });
    }

    @Nonnull
    public SelValue item(int i) {
        return match(new Match<SelValue>() {
            @Override
            public SelValue regular(SelValue[] array, SelType elementType) {
                return array[i];
            }

            @Override
            public SelValue doubles(double[] doubles) {
                return new SelValueDouble(doubles[i]);
            }
        });
    }

    @Override
    public SelType type() {
        return new SelTypeVector(elementType);
    }

    @Override
    public SelValueGraphData castToGraphData() {
        if (array != null && array.length == 1 && elementType.isGraphData()) {
            return array[0].castToGraphData();
        }

        return super.castToGraphData();
    }

    @Override
    public String format() {
        if (array != null) {
            return Arrays.toString(array);
        }
        return Arrays.toString(doubles);
    }

    public double[] doubleArray() {
        return match(new Match<double[]>() {
            @Override
            public double[] regular(SelValue[] array, SelType elementType) {
                throw new IllegalStateException("Not matched vector type: " + elementType + " != " + SelTypes.DOUBLE);
            }

            @Override
            public double[] doubles(double[] doubles) {
                return doubles;
            }
        });
    }

    public SelValue[] valueArray() {
        return match(new Match<SelValue[]>() {
            @Override
            public SelValue[] regular(SelValue[] array, SelType elementType) {
                return array;
            }

            @Override
            public SelValue[] doubles(double[] doubles) {
                throw new IllegalStateException("cannot get double array");
            }
        });
    }

    private interface Match<R> {
        R regular(SelValue[] array, SelType elementType);
        R doubles(double[] doubles);
    }

    public <R> R match(Match<R> match) {
        if (array != null) {
            return match.regular(array, elementType);
        } else if (doubles != null) {
            return match.doubles(doubles);
        } else {
            throw new AssertionError("not typed array and not double array");
        }
    }

    @Override
    public String formatSummary() {
        return match(new Match<String>() {
            @Override
            public String regular(SelValue[] array, SelType elementType) {
                return ListFormatShort.formatList(array.length, i -> array[i].formatSummary());
            }

            @Override
            public String doubles(double[] doubles) {
                return ListFormatShort.formatList(doubles.length, i -> Double.toString(doubles[i]));
            }
        });
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        SelValueVector that = (SelValueVector) o;

        if (!Arrays.equals(array, that.array)) return false;
        if (!Arrays.equals(doubles, that.doubles)) return false;
        return Objects.equals(elementType, that.elementType);
    }

    @Override
    public int hashCode() {
        int result = Arrays.hashCode(array);
        result = 31 * result + Arrays.hashCode(doubles);
        result = 31 * result + (elementType != null ? elementType.hashCode() : 0);
        return result;
    }
}
