package ru.yandex.solomon.expression.value;

import java.util.function.BiFunction;
import java.util.function.DoubleUnaryOperator;
import java.util.function.Function;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.math.doubles.DoubleMath;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.GraphData;

/**
 * @author Vladimir Gordiychuk
 */
public final class SelValues {
    private SelValues() {
    }

    public static SelValueGraphData getSingleGraphData(SelValueWithRange sourceWithRange) {
        SelValue source = sourceWithRange.getValue();
        PositionRange range = sourceWithRange.getRange();
        if (source.type().isVector() && source.type().vector().elementType.isGraphData()) {
            SelValueVector selValueVector = source.castToVector();

            if (selValueVector.length() == 0) {
                throw new EvaluationException(range, "No metrics found in vector of graph data");
            }

            if (selValueVector.length() > 1) {
                throw new EvaluationException(range, "Too many metrics found in vector of graph data: " +
                    selValueVector.length() + ", expected single metric");
            }

            return selValueVector.item(0).castToGraphData();
        }

        if (source.type().isGraphData()) {
            return source.castToGraphData();
        }

        throw new EvaluationException(range,
            "Unsupported argument type: " + source.type() + ", expected graph data or vector of graph data");
    }

    public static SelValue mapToGraphData(SelValueWithRange sourceWithRange, Function<GraphData, GraphData> fn) {
        SelValue source = sourceWithRange.getValue();
        PositionRange range = sourceWithRange.getRange();
        if (source.type().isGraphData()) {
            SelValueGraphData value = source.castToGraphData();
            NamedGraphData ngd = value.getNamedGraphData();
            GraphData gd = ngd.getGraphData();
            NamedGraphData newGraphData = ngd.toBuilder()
                .setGraphData(fn.apply(gd))
                .build();
            return new SelValueGraphData(newGraphData);
        } else if (source.type().isVector() && source.type().vector().elementType.isGraphData()) {
            SelValueGraphData[] items = Stream.of(source.castToVector().valueArray())
                    .map(SelValue::castToGraphData)
                    .map(value -> {
                        NamedGraphData ngd = value.getNamedGraphData();
                        GraphData gd = ngd.getGraphData();
                        NamedGraphData newGraphData = ngd.toBuilder()
                            .setGraphData(fn.apply(gd))
                            .build();
                        return new SelValueGraphData(newGraphData);
                    })
                    .toArray(SelValueGraphData[]::new);

            return new SelValueVector(SelTypes.GRAPH_DATA, items);
        }

        throw new EvaluationException(range, "Unsupported argument type: " + source.type());
    }

    public static SelValue mapToGraphData(
            SelValueWithRange sourceWithRange,
            SelValueWithRange helperWithRange,
            BiFunction<GraphData, GraphData, GraphData> fn)
    {
        SelValue source = sourceWithRange.getValue();
        PositionRange sourceRange = sourceWithRange.getRange();
        SelValue helper = helperWithRange.getValue();
        PositionRange helperRange = helperWithRange.getRange();
        if ((source.type().isGraphData() && helper.type().isGraphData())
            || (source.type().isGraphData() && helper.type().isGraphDataVector())
            || (source.type().isGraphDataVector() && helper.type().isGraphData()))
        {
            SelValueGraphData value = getSingleGraphData(sourceWithRange);
            SelValueGraphData valueHelper = getSingleGraphData(helperWithRange);
            NamedGraphData ngd = value.getNamedGraphData();
            GraphData gd = ngd.getGraphData();
            GraphData gdh = valueHelper.getGraphData();
            GraphData newGraphData = fn.apply(gd, gdh);
            NamedGraphData newNamedGraphData = ngd.toBuilder()
                .setGraphData(newGraphData)
                .build();
            return new SelValueGraphData(newNamedGraphData);
        } else if (source.type().isGraphDataVector() && helper.type().isGraphDataVector()) {
            SelValue[] sourceVec = source.castToVector().valueArray();
            SelValue[] helperVec = helper.castToVector().valueArray();
            if (sourceVec.length != helperVec.length) {
                throw new EvaluationException(PositionRange.convexHull(sourceRange, helperRange),
                    "Source and helper arrays have different size");
            }
            SelValueGraphData[] items = IntStream.range(0, sourceVec.length)
                .mapToObj(i -> {
                    NamedGraphData ngd = sourceVec[i].castToGraphData().getNamedGraphData();
                    GraphData gd = ngd.getGraphData();
                    GraphData gdh = helperVec[i].castToGraphData().getGraphData();
                    GraphData newGraphData = fn.apply(gd, gdh);
                    NamedGraphData newNamedGraphData = ngd.toBuilder()
                        .setGraphData(newGraphData)
                        .build();
                    return new SelValueGraphData(newNamedGraphData);
                })
                .toArray(SelValueGraphData[]::new);

            return new SelValueVector(SelTypes.GRAPH_DATA, items);
        }

        throw new EvaluationException(PositionRange.convexHull(sourceRange, helperRange),
            "Unsupported argument types: " + source.type() + " and " + helper.type());
    }

    public static SelValueDouble mapAsDouble(SelValueDouble value, DoubleUnaryOperator fn) {
        return new SelValueDouble(fn.applyAsDouble(value.getValue()));
    }

    public static SelValueVector mapArrayAsDouble(SelValueVector value, DoubleUnaryOperator fn) {
        return new SelValueVector(DoubleMath.map(value.doubleArray(), fn));
    }

    public static SelValueGraphData mapValueAsDouble(SelValueGraphData value, DoubleUnaryOperator fn) {
        var ngd = value.getNamedGraphData();
        var result = ngd.toBuilder()
            .setType(ru.yandex.monlib.metrics.MetricType.DGAUGE)
            .setGraphData(MetricType.DGAUGE, DoubleMath.map(ngd.getDataType(), ngd.getAggrGraphDataArrayList(), fn))
            .build();
        return new SelValueGraphData(result);
    }

    public static SelValueVector mapValuesAsDouble(SelValueVector value, DoubleUnaryOperator fn) {
        var source = value.castToVector();
        SelValueGraphData[] items = Stream.of(source.valueArray())
            .map(SelValue::castToGraphData)
            .map(v -> {
                var ngd = v.getNamedGraphData();
                var gd = ngd.getAggrGraphDataArrayList();
                var newGraphData = ngd.toBuilder()
                    .setType(ru.yandex.monlib.metrics.MetricType.DGAUGE)
                    .setGraphData(MetricType.DGAUGE, DoubleMath.map(ngd.getDataType(), gd, fn))
                    .build();
                return new SelValueGraphData(newGraphData);
            })
            .toArray(SelValueGraphData[]::new);
        return new SelValueVector(SelTypes.GRAPH_DATA, items);
    }
}
