package ru.yandex.solomon.flags;

import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

/**
 * @author Vladimir Gordiychuk
 */
public enum FeatureFlag {
    TEST("test", "test description"),
    EXAMPLE("example", "example description"),
    GRID_DOWNSAMPLING,
    // Truncated ts to specified grid during write
    GRID_ROUNDING,

    // Expression flags
    EXPRESSION_VECTORED_TYPES,
    EXPRESSION_LAST_VERSION,

    // Gateway uses Dataproxy
    METADATA_FROM_DATAPROXY,
    SERIES_FROM_DATAPROXY,

    // Alerting uses Dataproxy (no separate flags for meta/data)
    ALERTING_DATAPROXY,

    SYNC_AMBRY_ALERTS,

    // Check that project is internal cloud in cloud environments
    INTERNAL_CLOUD,

    // Resolve resource ids to names, see https://st.yandex-team.ru/SOLOMON-5984
    RESOURCE_NAME,

    // Use IAM authorizer for project instead of Project ACL, see https://st.yandex-team.ru/SOLOMON-6930
    USE_IAM_AUTHORIZER_ONLY,

    // Make authorize on push when in configs auth globally disabled
    AUTHORIZE_ON_PUSH_WHEN_GLOBALLY_DISABLED,
    BACKGROUND_CHECK_AUTH_ON_PUSH_WHEN_GLOBALLY_DISABLED,

    //TODO PM migration flags to remove

    // Use PM client instead of YdbProjectDao
    READ_PROJECT_FROM_PM("Use PM(projects)", "Use projects from project manager"),
    // Use PM authorizer
    USE_PM_AUTHORIZER("Use PM(auth)", "Use project manager authorization"),
    // Use PM Juggle integration
    USE_PM_JUGGLER_INTEGRATION("Push project changes", "Push project changes to juggler"),
    // Use PK for cloud updates
    DISABLE_PM_CLOUD_FETCHING("Fetch clouds disabled", "Fetch clouds by component disabled"),

    // End of PM flags

    DUPLICATE_TELEGRAM_TO_MESSENGER,

    DISABLE_GLOBAL_JUGGLER, // disables sending targetConfigs to Yandex's backend

    PARTITIONED_SHARD, // shard has several partitions

    USE_UNIFIED_AGENT_FOR_BILLING, //use unified agent instead of push client

    USE_CLOUD_LOGBROKER_FOR_BILLING,
    // put new flag between last flag and this comment
    ;

    private static final Map<String, FeatureFlag> FLAG_BY_NAME =
        Stream.of(FeatureFlag.values())
            .collect(Collectors.toMap(Enum::name, Function.identity()));
    private final String displayName;
    private final String description;

    FeatureFlag() {
        this("", "");
    }

    FeatureFlag(String displayName, String description) {
        this.displayName = displayName;
        this.description = description;
    }

    public String getDescription() {
        return description;
    }

    public String getDisplayName() {
        return displayName;
    }

    @Nullable
    public static FeatureFlag byName(String name) {
        return FLAG_BY_NAME.get(name);
    }
}
