package ru.yandex.solomon.flags;

import java.io.IOException;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.fasterxml.jackson.databind.node.ObjectNode;
import it.unimi.dsi.fastutil.ints.Int2BooleanArrayMap;

/**
 * @author Vladimir Gordiychuk
 */
@JsonSerialize(using = FeatureFlags.Serializer.class)
@JsonDeserialize(using = FeatureFlags.Deserializer.class)
public class FeatureFlags {
    public static final FeatureFlags EMPTY = new FeatureFlags() {
        @Override
        public void add(FeatureFlag flag, boolean value) {
            throw new IllegalStateException("Immutable");
        }

        @Override
        public void combine(FeatureFlags other) {
            throw new IllegalStateException("Immutable");
        }
    };

    private final Int2BooleanArrayMap flags;

    public FeatureFlags() {
        flags = new Int2BooleanArrayMap();
    }

    FeatureFlags(FeatureFlags copy) {
        this.flags = new Int2BooleanArrayMap(copy.flags);
    }

    void add(FeatureFlag flag, boolean value) {
        flags.put(flag.ordinal(), value);
    }

    public boolean isDefined(FeatureFlag flag) {
        return flags.containsKey(flag.ordinal());
    }

    public boolean hasFlag(FeatureFlag flag) {
        return flags.get(flag.ordinal());
    }

    void combine(@Nullable FeatureFlags other) {
        if (other == null || other.flags.isEmpty()) {
            return;
        }
        var it = other.flags.int2BooleanEntrySet().fastIterator();
        while (it.hasNext()) {
            var entry = it.next();
            int flag = entry.getIntKey();
            if (!flags.containsKey(flag)) {
                flags.put(flag, entry.getBooleanValue());
            }
        }
    }

    public boolean isEmpty() {
        return flags.isEmpty();
    }

    public boolean hasAnyFlag() {
        if (flags.isEmpty()) {
            return false;
        }

        var it = flags.values().iterator();
        while (it.hasNext()) {
            if (it.nextBoolean()) {
                return true;
            }
        }
        return false;
    }

    public boolean hasAnyFlag(boolean state) {
        if (flags.isEmpty()) {
            return false;
        }

        var it = flags.values().iterator();
        while (it.hasNext()) {
            if (it.nextBoolean() == state) {
                return true;
            }
        }
        return false;
    }

    public void clear() {
        flags.clear();
    }

    @Override
    public String toString() {
        var builder = new StringBuilder();
        builder.append("Flags{");
        var it = flags.int2BooleanEntrySet().fastIterator();
        while (it.hasNext()) {
            var entry = it.next();
            builder.append(FeatureFlag.values()[entry.getIntKey()])
                    .append("=")
                    .append(entry.getBooleanValue());
            if (it.hasNext()) {
                builder.append(", ");
            }
        }
        builder.append("}");
        return builder.toString();
    }

    public static class Serializer extends JsonSerializer<FeatureFlags> {
        @Override
        public void serialize(FeatureFlags value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
            gen.writeStartObject();
            var it = value.flags.int2BooleanEntrySet().fastIterator();
            while (it.hasNext()) {
                var entry = it.next();
                gen.writeBooleanField(FeatureFlag.values()[entry.getIntKey()].name(), entry.getBooleanValue());
            }
            gen.writeEndObject();
        }
    }

    public static class Deserializer extends JsonDeserializer<FeatureFlags> {
        @Override
        public FeatureFlags deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
            var node = (ObjectNode) p.readValueAsTree();
            var it = node.fieldNames();
            FeatureFlags result = new FeatureFlags();
            while (it.hasNext()) {
                var field = it.next();
                var flag = FeatureFlag.byName(field);
                if (flag == null) {
                    continue;
                }

                result.add(flag, node.get(field).booleanValue());
            }
            return result;
        }
    }
}
