package ru.yandex.solomon.flags;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * @author Vladimir Gordiychuk
 */
public class FeatureFlagsConfig {
    private static final FeatureFlags YASM_DEFAULT_FLAGS;
    static {
        var flags = new FeatureFlags();
        flags.add(FeatureFlag.SERIES_FROM_DATAPROXY, true);
        flags.add(FeatureFlag.GRID_DOWNSAMPLING, true);
        flags.add(FeatureFlag.METADATA_FROM_DATAPROXY, true);
        flags.add(FeatureFlag.EXPRESSION_LAST_VERSION, true);
        YASM_DEFAULT_FLAGS = flags;
    }

    @JsonProperty("default")
    private final FeatureFlags defaultFlags;
    @JsonProperty("project")
    private final Map<String, FeatureFlagsProject> flagsByProjectId;
    @JsonProperty("serviceProvider")
    private final Map<String, FeatureFlags> flagsByServiceProvider;
    @JsonProperty("createdAt")
    private final long createdAt;

    public FeatureFlagsConfig() {
        this.defaultFlags = new FeatureFlags();
        this.flagsByProjectId = new HashMap<>();
        this.flagsByServiceProvider = new HashMap<>();
        this.createdAt = System.currentTimeMillis();
    }

    public void addDefaultFlag(FeatureFlag flag, boolean value) {
        defaultFlags.add(flag, value);
    }

    public void addServiceProviderFlag(String service, FeatureFlag flag, boolean value) {
        var flags = flagsByServiceProvider.computeIfAbsent(service, ignore -> new FeatureFlags());
        flags.add(flag, value);
    }

    public void addProjectFlag(String projectId, FeatureFlag flag, boolean value) {
        var project = flagsByProjectId.get(projectId);
        if (project == null) {
            project = new FeatureFlagsProject(new FeatureFlags());
            flagsByProjectId.put(projectId, project);
        }
        project.addProjectFlag(flag, value);
    }

    public FeatureFlagsProject getProjectFlags(String projectId) {
        var result = flagsByProjectId.get(projectId);
        if (result == null) {
            result = new FeatureFlagsProject(FeatureFlags.EMPTY);
            flagsByProjectId.put(projectId, result);
        }
        return result;
    }

    public boolean isFlagsDefine(String projectId) {
        return isYasm(projectId) || flagsByProjectId.containsKey(projectId);
    }

    public FeatureFlags getDefaultFlags() {
        return defaultFlags;
    }

    public FeatureFlags getServiceProviderFlags(String serviceProviderId) {
        return flagsByServiceProvider.getOrDefault(serviceProviderId, FeatureFlags.EMPTY);
    }

    public FeatureFlags getFlags(String projectId, String shardId, String clusterId, String serviceId, String serviceProvider) {
        if (!flagsByProjectId.containsKey(projectId)
                && !flagsByServiceProvider.containsKey(serviceProvider)
                && !isYasm(projectId))
        {
            return defaultFlags;
        }

        var flags = new FeatureFlags();
        combineFlags(flags, projectId, shardId, clusterId, serviceId, serviceProvider);
        return flags;
    }

    public FeatureFlags getFlags(String projectId) {
        var project = flagsByProjectId.get(projectId);
        if (project == null && !isYasm(projectId)) {
            return defaultFlags;
        }
        var flags = new FeatureFlags();
        combineFlags(flags, projectId);
        return flags;
    }

    public FeatureFlags getFlags(String projectId, String serviceProvider) {
        if (!flagsByProjectId.containsKey(projectId)
                && !flagsByServiceProvider.containsKey(serviceProvider)
                && !isYasm(projectId))
        {
            return defaultFlags;
        }

        var flags = new FeatureFlags();
        var project = flagsByProjectId.get(projectId);
        if (project != null) {
            flags.combine(project.getDefaultFlags());
        }
        flags.combine(flagsByServiceProvider.getOrDefault(serviceProvider, FeatureFlags.EMPTY));
        if (isYasm(projectId)) {
            flags.combine(YASM_DEFAULT_FLAGS);
        }
        flags.combine(defaultFlags);
        return flags;
    }

    @Nullable
    public String define(FeatureFlag flag, String projectId, String shardId, String clusterId, String serviceId, String serviceProvider) {
        var project = flagsByProjectId.get(projectId);
        String define = null;
        if (project != null) {
            define = project.define(flag, shardId, clusterId, serviceId);
        }

        if (define == null && flagsByServiceProvider.getOrDefault(serviceProvider, FeatureFlags.EMPTY).isDefined(flag)) {
            define = "serviceProvider";
        }

        if (define == null && isYasm(projectId)) {
            define = YASM_DEFAULT_FLAGS.isDefined(flag) ? "yasm_default" : null;
        }

        if (define == null) {
            define = defaultFlags.isDefined(flag) ? "default" : null;
        }
        return define;
    }

    public void combineFlags(FeatureFlags flags, String projectId) {
        var project = flagsByProjectId.get(projectId);
        if (project != null) {
            flags.combine(project.getDefaultFlags());
        }
        if (isYasm(projectId)) {
            flags.combine(YASM_DEFAULT_FLAGS);
        }
        flags.combine(defaultFlags);
    }

    public void combineFlags(FeatureFlags flags, String projectId, String shardId, String clusterId, String serviceId, String serviceProviderId) {
        var project = flagsByProjectId.get(projectId);
        if (project != null) {
            project.combineFlags(flags, shardId, clusterId, serviceId);
        }
        flags.combine(flagsByServiceProvider.getOrDefault(serviceProviderId, FeatureFlags.EMPTY));
        if (isYasm(projectId)) {
            flags.combine(YASM_DEFAULT_FLAGS);
        }
        flags.combine(defaultFlags);
    }

    public long getCreatedAt() {
        return createdAt;
    }

    private boolean isYasm(String projectId) {
        return projectId.startsWith("yasm_");
    }

    @Override
    public String toString() {
        return "FeatureFlagsConfig{" +
                "defaultFlags=" + defaultFlags +
                ", flagsByServiceProvider=" + flagsByServiceProvider +
                ", flagsByProjectId=" + flagsByProjectId +
                '}';
    }
}
