package ru.yandex.solomon.flags;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * @author Vladimir Gordiychuk
 */
@JsonInclude(JsonInclude.Include.NON_EMPTY)
public class FeatureFlagsProject {
    @JsonProperty("default")
    private final FeatureFlags defaultFlags;
    @JsonProperty("cluster")
    private final Map<String, FeatureFlags> flagsByClusterId = new HashMap<>();
    @JsonProperty("service")
    private final Map<String, FeatureFlags> flagsByServiceId = new HashMap<>();
    @JsonProperty("shard")
    private final Map<String, FeatureFlags> flagsByShardId = new HashMap<>();

    public FeatureFlagsProject() {
        this(FeatureFlags.EMPTY);
    }

    public FeatureFlagsProject(FeatureFlags defaultFlags) {
        this.defaultFlags = defaultFlags;
    }

    public void addProjectFlag(FeatureFlag flag, boolean value) {
        defaultFlags.add(flag, value);
    }

    public void addServiceFlag(String id, FeatureFlag flag, boolean value) {
        addFlag(flagsByServiceId, id, flag, value);
    }

    public void addClusterFlag(String id, FeatureFlag flag, boolean value) {
        addFlag(flagsByClusterId, id, flag, value);
    }

    public void addShardFlag(String id, FeatureFlag flag, boolean value) {
        addFlag(flagsByShardId, id, flag, value);
    }

    public FeatureFlags getDefaultFlags() {
        return defaultFlags;
    }

    public void combineFlags(FeatureFlags flags, String shardId, String clusterId, String serviceId) {
        flags.combine(flagsByShardId.get(shardId));
        flags.combine(flagsByClusterId.get(clusterId));
        flags.combine(flagsByServiceId.get(serviceId));
        flags.combine(defaultFlags);
    }

    @Nullable
    public String define(FeatureFlag flag, String shardId, String clusterId, String serviceId) {
        if (isDefined(flag, flagsByShardId.get(shardId))) {
            return "shard";
        }

        if (isDefined(flag, flagsByClusterId.get(clusterId))) {
            return "cluster";
        }

        if (isDefined(flag, flagsByServiceId.get(serviceId))) {
            return "service";
        }

        if (isDefined(flag, defaultFlags)) {
            return "project";
        }

        return null;
    }

    private boolean isDefined(FeatureFlag flag, @Nullable FeatureFlags flags) {
        if (flags == null) {
            return false;
        }
        return flags.isDefined(flag);
    }

    private void addFlag(Map<String, FeatureFlags> flagById, String id, FeatureFlag flag, boolean value) {
        var flags = flagById.get(id);
        if (flags == null) {
            flags = new FeatureFlags();
            flagById.put(id, flags);
        }
        flags.add(flag, value);
    }

    @Override
    public String toString() {
        return "FeatureFlagsProject{" +
                "defaultFlags=" + defaultFlags +
                ", flagsByClusterId=" + flagsByClusterId +
                ", flagsByServiceId=" + flagsByServiceId +
                ", flagsByShardId=" + flagsByShardId +
                '}';
    }
}
