package ru.yandex.solomon.fetcher.client;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.cluster.discovery.ClusterDiscoveryImpl;
import ru.yandex.discovery.DiscoveryService;
import ru.yandex.fetcher.api.FetcherServiceGrpc;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto.ShardsHealthRequest;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto.ShardsHealthResponse;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto.TReloadShardRequest;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto.TResolveClusterRequest;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto.TResolveClusterResponse;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto.TargetsStatusRequest;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto.TargetsStatusResponse;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.util.host.HostUtils;


/**
 * @author Sergey Polovko
 */
public class FetcherClientGrpc implements FetcherClient, AutoCloseable {

    private final ClusterDiscovery<GrpcTransport> discovery;

    public FetcherClientGrpc(
            TGrpcClientConfig config,
            @Nullable String clientId,
            ThreadPoolProvider threadPoolProvider,
            DiscoveryService discovery,
            ClientOptionsFactory factory)
    {
        var options = factory.newBuilder("FetcherClientConfig", config)
            .setClientId(clientId)
            .build();
        var executor = threadPoolProvider.getExecutorService("CpuLowPriority", "");
        var timer = threadPoolProvider.getSchedulerExecutorService();
        this.discovery = new ClusterDiscoveryImpl<>(address -> {
            return new GrpcTransport(address, options);
        }, config.getAddressesList(), discovery, timer, executor, TimeUnit.HOURS.toMillis(1L));
    }

    @Override
    public Set<String> getKnownHosts() {
        return discovery.getNodes();
    }

    @Override
    public String getAnyAvailableHost() {
        final int maxHosts = 5;
        List<String> availableHosts = new ArrayList<>(maxHosts);

        for (var node : discovery.getNodes()) {
            var transport = discovery.getTransportByNode(node);
            if (transport != null && transport.isConnected()) {
                availableHosts.add(node);
                if (availableHosts.size() == maxHosts) {
                    break;
                }
            }
        }

        if (availableHosts.isEmpty()) {
            throw new IllegalStateException("there are no available Fetcher hosts");
        }

        int idx = ThreadLocalRandom.current().nextInt(availableHosts.size());
        return availableHosts.get(idx);
    }

    @Override
    public CompletableFuture<TargetsStatusResponse> targetsStatus(String host, TargetsStatusRequest request) {
        GrpcTransport transport = getTransport(host);
        if (transport == null) {
            return unknownHost(host);
        }
        return transport.unaryCall(FetcherServiceGrpc.getTargetStatusMethod(), request);
    }

    @Override
    public CompletableFuture<ShardsHealthResponse> getFetcherShardsHealths(String host) {
        GrpcTransport transport = getTransport(host);
        if (transport == null) {
            return unknownHost(host);
        }
        ShardsHealthRequest request = ShardsHealthRequest.getDefaultInstance();
        return transport.unaryCall(FetcherServiceGrpc.getShardsHealthMethod(), request);
    }

    @Override
    public CompletableFuture<TResolveClusterResponse> resolveCluster(String host, TResolveClusterRequest request) {
        GrpcTransport transport = getTransport(host);
        if (transport == null) {
            return unknownHost(host);
        }
        return transport.unaryCall(FetcherServiceGrpc.getResolveClusterMethod(), request);
    }

    @Override
    public CompletableFuture<Void> reloadShard(String host, String projectId, String shardId, int shardNumId) {
        GrpcTransport transport = getTransport(host);
        if (transport == null) {
            return unknownHost(host);
        }
        TReloadShardRequest request = TReloadShardRequest.newBuilder()
            .setProjectId(projectId)
            .setShardId(shardId)
            .setNumId(shardNumId)
            .build();
        return transport.unaryCall(FetcherServiceGrpc.getReloadShardMethod(), request)
            .thenAccept(r -> {});
    }

    @Nullable
    private GrpcTransport getTransport(String host) {
        GrpcTransport transport = discovery.getTransportByNode(host);
        if (transport != null) {
            return transport;
        }

        // try 'localhost' if the host is the same as the current one
        if (HostUtils.getFqdn().equals(host)) {
            return discovery.getTransportByNode("localhost");
        }

        return null;
    }

    private static <T> CompletableFuture<T> unknownHost(String host) {
        return CompletableFuture.failedFuture(new IllegalArgumentException("unknown host: " + host));
    }

    @Override
    public void close() {
        discovery.close();
    }
}
