package ru.yandex.grpc.conf;

import java.util.Optional;

import io.grpc.NameResolver;
import io.netty.resolver.dns.DnsNameResolverBuilder;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.grpc.Grpc;
import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.grpc.utils.LimiterOptions;
import ru.yandex.grpc.utils.NettyDnsNameResolver;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.passport.tvmauth.TvmClient;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;

import static ru.yandex.solomon.config.OptionalSet.setBytesSize;
import static ru.yandex.solomon.util.NettyUtils.clientDatagramChannelForEventLoop;

/**
 * @author Alexei Suleimanov
 */
public class ClientOptionsFactory {
    private final Optional<TvmClient> tvmClientOptional;
    private final Optional<TokenProvider> iamTokenProvider;
    private final ThreadPoolProvider threadPoolProvider;

    public ClientOptionsFactory(
            Optional<TvmClient> tvmClientOptional,
            Optional<TokenProvider> iamTokenProvider,
            ThreadPoolProvider threadPoolProvider)
    {
        this.tvmClientOptional = tvmClientOptional;
        this.iamTokenProvider = iamTokenProvider;
        this.threadPoolProvider = threadPoolProvider;
    }

    public DefaultClientOptions.Builder newBuilder(
            String configPrefix,
            TGrpcClientConfig config) {
        DefaultClientOptions.Builder builder = DefaultClientOptions.newBuilder();

        setBytesSize(builder::setMaxInboundMessageSizeInBytes, config.getMaxInboundMessageSize());
        setBytesSize(builder::setMaxOutboundMessageSizeInBytes, config.getMaxOutboundMessageSize());

        if (config.hasCircuitBreakerConfig()) {
            var circuitBreakerConfig = config.getCircuitBreakerConfig();
            builder.setCircuitBreakerFailureQuantileThreshold(circuitBreakerConfig.getFailureQuantileThreshold());

            var timeout = circuitBreakerConfig.getResetTimeout();
            builder.setCircuitBreakerResetTimeout(timeout.getValue(), TimeUnitConverter.protoToUnit(timeout.getUnit()));
        }

        for (var limiterConfig : config.getLimiterConfigList()) {
            builder.addLimiterOptions(
                    LimiterOptions.newBuilder()
                            .setFromConfig(limiterConfig)
                            .build());
        }

        var executorService = threadPoolProvider.getExecutorService(
                config.getThreadPoolName(),
                configPrefix + ".GrpcConfig.ThreadPoolName");

        builder.setTimer(threadPoolProvider.getSchedulerExecutorService());
        builder.setRpcExecutor(executorService);
        builder.setResponseHandlerExecutorService(executorService);
        builder.setEventLoopGroup(threadPoolProvider.getIOExecutor());

        if (config.hasIdleTimeout()) {
            var timeout = config.getIdleTimeout();
            builder.setIdleTimeOut(timeout.getValue(), TimeUnitConverter.protoToUnit(timeout.getUnit()));
        }

        if (config.hasKeepAliveTime()) {
            var time = config.getKeepAliveTime();
            builder.setKeepAliveDelay(time.getValue(), TimeUnitConverter.protoToUnit(time.getUnit()));
        }

        if (config.hasKeepAliveTimeout()) {
            var timeout = config.getKeepAliveTimeout();
            builder.setKeepAliveTimeout(timeout.getValue(), TimeUnitConverter.protoToUnit(timeout.getUnit()));
        }

        if (config.hasReadTimeout()) {
            var timeout = config.getReadTimeout();
            builder.setRequestTimeOut(timeout.getValue(), TimeUnitConverter.protoToUnit(timeout.getUnit()));
        }

        if (!config.getDisableNettyDnsNameResolver()) {
            builder.setNameResolverFactory(createNettyDnsNameResolver(threadPoolProvider));
        }

        builder.setUseTls(config.getUseTls());


        switch (config.getAuthCase()) {
            case AUTHIAM -> {
                if (iamTokenProvider.isEmpty()) {
                    throw new IllegalStateException("Iam token provider is not configured");
                }
                builder.setCallCredentialsSupplier(() -> Grpc.authCredentials(
                        AuthType.IAM.getMetadataKey(),
                        AuthType.IAM.getValuePrefix(),
                        () -> iamTokenProvider.get().getToken()));
            }
            case AUTHTVM -> {
                if (tvmClientOptional.isEmpty()) {
                    throw new IllegalStateException("Tvm client is not configured, destination client: " + config.getAuthTvm().getDestinationClientId());
                }
                builder.setCallCredentialsSupplier(() -> Grpc.authCredentials(
                        AuthType.TvmService.getMetadataKey(),
                        AuthType.TvmService.getValuePrefix(),
                        () -> tvmClientOptional.get().getServiceTicketFor(config.getAuthTvm().getDestinationClientId())));
            }
        }
        return builder;
    }

    private static NameResolver.Factory createNettyDnsNameResolver(ThreadPoolProvider threadPoolProvider) {
        var timer = threadPoolProvider.getSchedulerExecutorService();
        var elg = threadPoolProvider.getIOExecutor();
        var resolver = new DnsNameResolverBuilder(elg.next())
                .channelType(clientDatagramChannelForEventLoop(elg))
                .build();

        return NettyDnsNameResolver.newFactory(resolver, timer, MetricRegistry.root());
    }
}
