package ru.yandex.solomon.grpc.handler;

import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Context;
import io.grpc.stub.StreamObserver;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.exception.handlers.GrpcApiExceptionResolver;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public final class GrpcMethodHandler {

    public static <ReqT, RespT> void handle(
            HandlerFunc<ReqT, RespT> handler,
            ReqT request,
            StreamObserver<RespT> responseObserver) {
        asyncComplete(
                CompletableFutures.safeCall(() -> {
                    AuthSubject subject = AuthenticationInterceptor.getAuthSubject(Context.current());
                    return handler.apply(request, subject);
                })
                        .exceptionally(t -> {
                            throw GrpcApiExceptionResolver.doResolveException(t);
                        }),
                responseObserver
        );
    }

    public static <ReqT, RespT, ServiceResp> void handle(
            HandlerFunc<ReqT, ServiceResp> handler,
            ReqT request,
            StreamObserver<RespT> responseObserver,
            BiFunction<ServiceResp, Throwable, RespT> resultHandler) {
        asyncComplete(
                CompletableFutures.safeCall(() -> {
                    AuthSubject subject = AuthenticationInterceptor.getAuthSubject(Context.current());
                    return handler.apply(request, subject)
                            .handle(resultHandler);
                })
                        .exceptionally(t -> {
                            throw GrpcApiExceptionResolver.doResolveException(t);
                        }),
                responseObserver
        );
    }

    public static <ReqT, RespT, ServiceResp> void handle(
            HandlerFunc<ReqT, ServiceResp> handler,
            ReqT request,
            StreamObserver<RespT> responseObserver,
            BiFunction<ServiceResp, Throwable, RespT> resultHandler,
            Supplier<AuthSubject> subjectSupplier) {
        asyncComplete(
                CompletableFutures.safeCall(() -> {
                    AuthSubject subject = subjectSupplier.get();
                    return handler.apply(request, subject)
                            .handle(resultHandler);
                })
                        .exceptionally(t -> {
                            throw GrpcApiExceptionResolver.doResolveException(t);
                        }),
                responseObserver
        );
    }

    public static <ReqT, RespT> void handle(
            SimpleHandlerFunc<ReqT, RespT> handler,
            ReqT request,
            StreamObserver<RespT> responseObserver) {
        try {
            AuthSubject subject = AuthenticationInterceptor.getAuthSubject(Context.current());
            responseObserver.onNext(handler.apply(request, subject));
            responseObserver.onCompleted();
        } catch (Exception e) {
            responseObserver.onError(GrpcApiExceptionResolver.doResolveException(e));
        }
    }

    public static <ReqT, RespT> void handle(
            SimpleHandlerFuncWithoutAuth<ReqT, RespT> handler,
            ReqT request,
            StreamObserver<RespT> responseObserver) {
        try {
            responseObserver.onNext(handler.apply(request));
            responseObserver.onCompleted();
        } catch (Exception e) {
            responseObserver.onError(GrpcApiExceptionResolver.doResolveException(e));
        }
    }

    public static <ReqT, RespT> void handleGet(
            GetHandlerFunc<ReqT, RespT> handler,
            ReqT request,
            StreamObserver<RespT> responseObserver) {
        asyncComplete(
                CompletableFutures.safeCall(() -> {
                    AuthSubject subject = AuthenticationInterceptor.getAuthSubject(Context.current());
                    return handler.apply(request, subject).thenApply(Pair::getLeft);
                })
                        .exceptionally(t -> {
                            throw GrpcApiExceptionResolver.doResolveException(t);
                        }),
                responseObserver
        );
    }

    public static <ReqT, RespT> void handleUpdate(
            UpdateHandlerFunc<ReqT, RespT> handler,
            ReqT request,
            StreamObserver<RespT> responseObserver) {
        asyncComplete(
                CompletableFutures.safeCall(() -> {
                    Context context = Context.current();
                    AuthSubject subject = AuthenticationInterceptor.getAuthSubject(context);
                    return handler.apply(request, subject, -2);
                })
                        .exceptionally(t -> {
                            throw GrpcApiExceptionResolver.doResolveException(t);
                        }),
                responseObserver
        );
    }

    public static <ReqT, ServiceResp, RespT> void handleUpdate(
            UpdateHandlerFunc<ReqT, ServiceResp> handler,
            ReqT request,
            StreamObserver<RespT> responseObserver,
            BiFunction<ServiceResp, Throwable, RespT> resultHandler) {
        asyncComplete(
                CompletableFutures.safeCall(() -> {
                    Context context = Context.current();
                    AuthSubject subject = AuthenticationInterceptor.getAuthSubject(context);
                    return handler.apply(request, subject, -2)
                            .handle(resultHandler);
                })
                        .exceptionally(t -> {
                            throw GrpcApiExceptionResolver.doResolveException(t);
                        }),
                responseObserver
        );
    }

    @FunctionalInterface
    public interface HandlerFunc<ReqT, RespT> {
        CompletableFuture<RespT> apply(ReqT request, AuthSubject subject);
    }

    @FunctionalInterface
    public interface GetHandlerFunc<ReqT, RespT> {
        CompletableFuture<Pair<RespT, Integer>> apply(ReqT request, AuthSubject subject);
    }

    @FunctionalInterface
    public interface UpdateHandlerFunc<ReqT, RespT> {
        CompletableFuture<RespT> apply(ReqT request, AuthSubject subject, int etag);
    }

    @FunctionalInterface
    public interface SimpleHandlerFunc<ReqT, RespT> {
        RespT apply(ReqT request, AuthSubject subject);
    }

    @FunctionalInterface
    public interface SimpleHandlerFuncWithoutAuth<ReqT, RespT> {
        RespT apply(ReqT request);
    }
}
