package ru.yandex.grpc.utils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import com.google.common.base.Strings;
import io.grpc.CallCredentials;
import io.grpc.NameResolver;
import io.grpc.internal.GrpcUtil;
import io.netty.channel.EventLoopGroup;

import ru.yandex.monlib.metrics.labels.validate.StrictValidator;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.util.host.HostUtils;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public final class DefaultClientOptions implements GrpcClientOptions {

    private static final DefaultClientOptions EMPTY = newBuilder().build();

    private final long idleTimeoutMs;
    private final long defaultTimeoutMs;
    private final long keepAliveDelayMs;
    private final long keepAliveTimeoutMs;
    private final int maxInboundMessageSizeInBytes;
    private final int maxOutboundMessageSizeInBytes;

    private final boolean useTls;

    @Nullable
    private final Executor rpcExecutor;
    @Nullable
    @WillNotClose
    private final ScheduledExecutorService timer;
    @Nullable
    private final EventLoopGroup eventLoopGroup;

    @Nullable
    @WillNotClose
    // TODO: remove this executor, use rpcExecutor only
    private final ExecutorService responseHandlerExecutorService;
    @Nullable
    private final ChannelFactory channelFactory;
    @Nullable
    private final CircuitBreakerFactory circuitBreakerFactory;
    private final double circuitBreakerFailureQuantileThreshold;
    private final long circuitBreakerResetTimeoutMillis;
    private final MetricRegistry metricRegistry;
    private final String clientId;
    @Nullable
    private final NameResolver.Factory nameResolverFactory;
    @Nullable
    private final Supplier<CallCredentials> callCredentialsSupplier;

    private final List<LimiterOptions> limiterOptions;
    private final Map<String, Long> maxInFlightForEndpoint;

    private DefaultClientOptions(Builder builder) {
        this.idleTimeoutMs = builder.idleTimeoutMs;
        this.defaultTimeoutMs = builder.defaultTimeoutMs;
        this.keepAliveDelayMs = builder.keepAliveDelayMs;
        this.keepAliveTimeoutMs = builder.keepAliveTimeoutMs;
        this.maxInboundMessageSizeInBytes = builder.maxInboundMessageSizeInBytes;
        this.maxOutboundMessageSizeInBytes = builder.maxOutboundMessageSizeInBytes;

        this.rpcExecutor = builder.rpcExecutor;
        this.timer = builder.timer;
        this.eventLoopGroup = builder.eventLoopGroup;
        this.responseHandlerExecutorService = builder.responseHandlerExecutorService;
        this.channelFactory = builder.channelFactory;
        this.useTls = builder.useTls;

        this.circuitBreakerFactory = builder.circuitBreakerFactory;
        this.circuitBreakerFailureQuantileThreshold = builder.circuitBreakerFailureQuantileThreshold;
        this.circuitBreakerResetTimeoutMillis = builder.circuitBreakerResetTimeoutMillis;
        this.metricRegistry = builder.metricRegistry != null
            ? builder.metricRegistry
            : MetricRegistry.root();
        this.clientId = Strings.isNullOrEmpty(builder.clientId)
            ? HostUtils.getShortName()
            : builder.clientId;
        this.nameResolverFactory = builder.nameResolverFactory;
        this.callCredentialsSupplier = builder.callCredentialsSupplier;
        this.maxInFlightForEndpoint = Map.copyOf(builder.maxInFlightForEndpoint);
        this.limiterOptions = List.copyOf(builder.limiterOptions);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static DefaultClientOptions empty() {
        return EMPTY;
    }

    @Override
    public boolean isUseTls() {
        return this.useTls;
    }

    @Override
    public long getIdleTimeoutMillis() {
        return idleTimeoutMs;
    }

    @Override
    public long getDefaultTimeoutMillis() {
        return defaultTimeoutMs;
    }

    @Override
    public int getMaxInboundMessageSizeInBytes() {
        return maxInboundMessageSizeInBytes;
    }

    @Override
    public int getMaxOutboundMessageSizeInBytes() {
        return maxOutboundMessageSizeInBytes;
    }

    @Override
    public long getKeepAliveDelayMillis() {
        return keepAliveDelayMs;
    }

    @Override
    public long getKeepAliveTimeoutMillis() {
        return keepAliveTimeoutMs;
    }

    @Override
    public Optional<EventLoopGroup> getEventLoopGroup() {
        return Optional.ofNullable(eventLoopGroup);
    }

    @Override
    public Optional<Executor> getRpcExecutor() {
        return Optional.ofNullable(rpcExecutor);
    }

    @Override
    public Optional<ScheduledExecutorService> getTimer() {
        return Optional.ofNullable(timer);
    }

    @Override
    public Optional<ExecutorService> getResponseHandlerExecutorService() {
        return Optional.ofNullable(responseHandlerExecutorService);
    }

    @Override
    public Optional<ChannelFactory> getChannelFactory() {
        return Optional.ofNullable(channelFactory);
    }

    @Override
    public Optional<NameResolver.Factory> getNameResolverFactory() {
        return Optional.ofNullable(nameResolverFactory);
    }

    @Override
    public Optional<Supplier<CallCredentials>> getCallCredentialsSupplier() {
        return Optional.ofNullable(callCredentialsSupplier);
    }

    @Override
    public double getCircuitBreakerFailureQuantileThreshold() {
        return circuitBreakerFailureQuantileThreshold;
    }

    @Override
    public long getCircuitBreakerResetTimeoutMillis() {
        return circuitBreakerResetTimeoutMillis;
    }

    @Override
    public Optional<CircuitBreakerFactory> getCircuitBreakerFactory() {
        return Optional.ofNullable(circuitBreakerFactory);
    }

    @Override
    public MetricRegistry getMetricRegistry() {
        return metricRegistry;
    }

    @Override
    public String getClientId() {
        return clientId;
    }

    @Override
    public long getInFlightLimitForEndpoint(String endpoint) {
        return maxInFlightForEndpoint.getOrDefault(endpoint, 0L);
    }

    @Override
    public List<LimiterOptions> getLimiterOptions() {
        return limiterOptions;
    }

    /**
     * BUILDER
     */
    @ParametersAreNonnullByDefault
    public static final class Builder implements GrpcClientOptions.Builder {
        private static final long DEFAULT_REQUEST_TIMEOUT = TimeUnit.SECONDS.toMillis(30L);
        private static final int DEFAULT_MAX_INBOUND_MESSAGE_SIZE_IN_BYTES = 32 << 20; // 32 MiB
        private static final int DEFAULT_MAX_OUTBOUND_MESSAGE_SIZE_IN_BYTES = 32 << 20; // 32 MiB
        private static final long DEFAULT_KEEPALIVE_TIME_MILLIS = TimeUnit.MINUTES.toMillis(5L);
        private static final long DEFAULT_KEEPALIVE_TIMEOUT_MILLIS = TimeUnit.SECONDS.toMillis(20L);

        private long idleTimeoutMs = 0; // use default value from gRPC
        private long defaultTimeoutMs = DEFAULT_REQUEST_TIMEOUT;
        private int maxInboundMessageSizeInBytes = DEFAULT_MAX_INBOUND_MESSAGE_SIZE_IN_BYTES;
        private int maxOutboundMessageSizeInBytes = DEFAULT_MAX_OUTBOUND_MESSAGE_SIZE_IN_BYTES;
        private long keepAliveDelayMs = DEFAULT_KEEPALIVE_TIME_MILLIS;
        private long keepAliveTimeoutMs = DEFAULT_KEEPALIVE_TIMEOUT_MILLIS;

        @Nullable
        private Executor rpcExecutor;

        @Nullable
        private ScheduledExecutorService timer;

        @Nullable
        private EventLoopGroup eventLoopGroup;

        @Nullable
        private ExecutorService responseHandlerExecutorService;

        @Nullable
        private ChannelFactory channelFactory;

        @Nullable
        private CircuitBreakerFactory circuitBreakerFactory;

        @Nullable
        private MetricRegistry metricRegistry;

        @Nullable
        private NameResolver.Factory nameResolverFactory;

        @Nullable
        private Supplier<CallCredentials> callCredentialsSupplier;

        // disabled by default
        private double circuitBreakerFailureQuantileThreshold = 0.0d;
        private long circuitBreakerResetTimeoutMillis = 0;
        private String clientId = "";

        private final Map<String, Long> maxInFlightForEndpoint = new HashMap<>();
        private final List<LimiterOptions> limiterOptions = new ArrayList<>();

        private boolean useTls;

        private Builder() {
        }

        /**
         * Executor service that handle rpc calls.
         * <p>
         * It's an optional parameter. If the user has not provided an executor when the client is
         * built, the builder will use a static cached thread pool.
         * <p>
         * The client won't take ownership of the given executor. It's caller's responsibility to
         * shut down the executor when it's desired.
         */
        public Builder setRpcExecutor(Executor executor) {
            this.rpcExecutor = executor;
            return this;
        }

        public Builder setTimer(ScheduledExecutorService timer) {
            this.timer = timer;
            return this;
        }

        public Builder setUseTls(boolean value) {
            this.useTls = value;
            return this;
        }

        /**
         * Provides an EventGroupLoop to be used by the netty transport.
         * <p>
         * The client won't take ownership of the given executor. It's caller's responsibility to
         * shut down the executor when it's desired.
         */
        public Builder setEventLoopGroup(EventLoopGroup eventLoopGroup) {
            this.eventLoopGroup = eventLoopGroup;
            return this;
        }

        /**
         * Set the duration without ongoing RPCs before going to idle mode.
         *
         * <p>In idle mode the channel shuts down all connections, the NameResolver and the
         * LoadBalancer. A new RPC would take the channel out of idle mode. A channel starts in idle mode.
         */
        public Builder setIdleTimeOut(long value, TimeUnit unit) {
            this.idleTimeoutMs = unit.toMillis(value);
            return this;
        }

        /**
         * Default timeout for each request to server.
         * <p>
         * It's an optional parameter. Default value is 30s.
         */
        public Builder setRequestTimeOut(long value, TimeUnit unit) {
            this.defaultTimeoutMs = unit.toMillis(value);
            return this;
        }

        /**
         * Sets the maximum message size allowed to be received on the channel. If not called,
         * defaults to 32 MiB. The default provides protection to clients who haven't considered the
         * possibility of receiving large messages while trying to be large enough to not be hit in normal
         * usage.
         *
         * @param bytesSize size in bytes
         */
        public Builder setMaxInboundMessageSizeInBytes(int bytesSize) {
            checkArgument(bytesSize >= 0, "bytesSize: " + bytesSize + " (expected: >= 0)");
            this.maxInboundMessageSizeInBytes = bytesSize;
            return this;
        }

        /**
         * Sets the maximum message size allowed to be send on the channel. If not called,
         * defaults to 32 MiB.
         *
         * @param bytesSize size in bytes
         */
        public Builder setMaxOutboundMessageSizeInBytes(int bytesSize) {
            checkArgument(bytesSize >= 0, "bytesSize: " + bytesSize + " (expected: >= 0)");
            this.maxOutboundMessageSizeInBytes = bytesSize;
            return this;
        }

        /**
         * Executor service that handle response from server, also reuse for retry requests.
         * <p>
         * It's an optional parameter. If the user has not provided an executor when the client is
         * built, the builder will use a cached thread pool.
         * <p>
         * The client won't take ownership of the given executor. It's caller's responsibility to
         * shut down the executor when it's desired.
         */
        public Builder setResponseHandlerExecutorService(ExecutorService responseHandlerExecutorService) {
            this.responseHandlerExecutorService = responseHandlerExecutorService;
            return this;
        }

        /**
         * When keep alive enabled define delay between keep alive messages.
         * If parameter not specified will be used default delay from gRPC
         * {@link GrpcUtil#DEFAULT_KEEPALIVE_TIMEOUT_NANOS}
         *
         * @see Builder#setKeepAliveTimeout(long, TimeUnit)
         */
        public Builder setKeepAliveDelay(long time, TimeUnit unit) {
            this.keepAliveDelayMs = unit.toMillis(time);
            return this;
        }

        /**
         * When keep alive enabled define timeout for each keep alive message.
         * If parameter not specified will be use default timeout from gRPC
         * {@link GrpcUtil#DEFAULT_KEEPALIVE_TIMEOUT_NANOS}
         *
         * @see Builder#setKeepAliveDelay(long, TimeUnit)
         */
        public Builder setKeepAliveTimeout(long time, TimeUnit unit) {
            this.keepAliveTimeoutMs = unit.toMillis(time);
            return this;
        }

        public Builder setChannelFactory(ChannelFactory channelFactory) {
            this.channelFactory = channelFactory;
            return this;
        }

        /**
         * Percent of failure before open circuit breaker. 0 disable circuit breaker at all. By default
         * 0.4
         *
         * @param quantile 0..1
         * @see ru.yandex.solomon.selfmon.failsafe.CircuitBreaker
         */
        public Builder setCircuitBreakerFailureQuantileThreshold(double quantile) {
            this.circuitBreakerFailureQuantileThreshold = quantile;
            return this;
        }

        /**
         * How much time CircuitBreaker will be open and reject all request before move to
         * half-open state. By default 30 second.
         *
         * @see ru.yandex.solomon.selfmon.failsafe.CircuitBreaker
         */
        public Builder setCircuitBreakerResetTimeout(long time, TimeUnit unit) {
            this.circuitBreakerResetTimeoutMillis = unit.toMillis(time);
            return this;
        }

        public Builder setCircuitBreakerFactory(CircuitBreakerFactory factory) {
            this.circuitBreakerFactory = factory;
            return this;
        }

        public Builder setMetricRegistry(MetricRegistry metricRegistry) {
            this.metricRegistry = metricRegistry;
            return this;
        }

        public Builder setNameResolverFactory(NameResolver.Factory nameResolverFactory) {
            this.nameResolverFactory = nameResolverFactory;
            return this;
        }

        public Builder setCallCredentialsSupplier(@Nullable Supplier<CallCredentials> callCredentialsSupplier) {
            this.callCredentialsSupplier = callCredentialsSupplier;
            return this;
        }

        /**
         * Client identity to identify the client to the server.
         */
        public Builder setClientId(@Nullable String clientId) {
            if (Strings.isNullOrEmpty(clientId)) {
                this.clientId = "";
                return this;
            }

            StrictValidator.SELF.checkValueValid(clientId);
            this.clientId = clientId;
            return this;
        }

        public Builder setMaxInflightForEndpoint(String endpoint, long maxInflight) {
            maxInFlightForEndpoint.put(endpoint, maxInflight);
            return this;
        }

        public Builder addLimiterOptions(LimiterOptions limiterOptions) {
            this.limiterOptions.add(limiterOptions);
            return this;
        }

        @Override
        public DefaultClientOptions build() {
            return new DefaultClientOptions(this);
        }
    }
}
