package ru.yandex.grpc.utils;

import java.net.InetSocketAddress;
import java.util.Arrays;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import com.google.common.base.Preconditions;
import com.google.protobuf.TextFormat;
import io.grpc.BindableService;
import io.grpc.Server;
import io.grpc.ServerInterceptor;
import io.grpc.netty.NettyServerBuilder;
import io.grpc.protobuf.services.ProtoReflectionService;
import io.netty.buffer.ByteBufAllocator;
import io.netty.channel.ChannelOption;
import io.opentracing.contrib.grpc.TracingServerInterceptor;

import ru.yandex.grpc.utils.server.ServerMetrics;
import ru.yandex.grpc.utils.server.interceptors.ClientFilterServerInterceptor;
import ru.yandex.grpc.utils.server.interceptors.MetricServerInterceptor;
import ru.yandex.grpc.utils.server.interceptors.MetricServerStreamTracer;
import ru.yandex.solomon.config.DataSizeConverter;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcServerConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.NettyUtils;

import static ru.yandex.solomon.config.OptionalSet.setInt;
import static ru.yandex.solomon.config.OptionalSet.setTime;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcServerFactory {
    private final ThreadPoolProvider threadpool;
    private final ServerMetrics metrics;
    private final ClientFilterServerInterceptor filter;

    public GrpcServerFactory(ThreadPoolProvider threadpool, ServerMetrics metrics, ClientFilterServerInterceptor filter) {
        this.threadpool = threadpool;
        this.metrics = metrics;
        this.filter = filter;
    }

    public Server makeServer(String cfgPrefix, TGrpcServerConfig config, BindableService... services) {
        return makeServer(cfgPrefix, config, Arrays.asList(services));
    }

    public Server makeServer(String cfgPrefix, TGrpcServerConfig config, Iterable<? extends BindableService> services) {
        NettyServerBuilder builder = newBuilder(cfgPrefix, config, services);
        builder.intercept(filter);
        builder.intercept(new MetricServerInterceptor());
        return builder.build();
    }

    public void startWithTimeout(TGrpcServerConfig config, Server server) {
        var future = asyncServerStart(config, server);
        cancelByTimeout(future, 5, TimeUnit.SECONDS);
    }

    private Future<?> asyncServerStart(TGrpcServerConfig config, Server server) {
        var executor = threadpool.getExecutorService(config.getThreadPoolName(), "CpuLowPriority");
        return executor.submit(() -> {
            try {
                server.start();
            } catch (Throwable e) {
                ExceptionUtils.uncaughtException(new RuntimeException("Failed start grpc server by config " + TextFormat.shortDebugString(config), e));
            }
        });
    }

    private void cancelByTimeout(Future<?> future, long value, TimeUnit unit) {
        var timer = threadpool.getSchedulerExecutorService();
        timer.schedule(() -> {
            try {
                if (!future.isDone()) {
                    future.cancel(true);
                    future.get(1, TimeUnit.SECONDS);
                }
            } catch (Throwable e) {
                ExceptionUtils.uncaughtException(new RuntimeException("Failed start grpc server in time", e));
            }
        }, value, unit);
    }

    public Server makePublicServer(
        String cfgPrefix,
        TGrpcServerConfig config,
        Iterable<? extends ServerInterceptor> interceptors,
        Iterable<? extends BindableService> services)
    {
        NettyServerBuilder builder = newBuilder(cfgPrefix, config, services);
        for (var interceptor : interceptors) {
            builder.intercept(interceptor);
        }
        return builder.build();
    }

    private NettyServerBuilder newBuilder(String cfgPrefix, TGrpcServerConfig config, Iterable<? extends BindableService> services) {
        ExecutorService executor = threadpool.getExecutorService(
            config.getThreadPoolName(),
            cfgPrefix + ".ThreadPoolName");
        Preconditions.checkArgument(config.getPortCount() > 0, "empty list of ports in " + cfgPrefix);
        NettyServerBuilder builder = NettyServerBuilder.forPort(config.getPort(0))
            .executor(executor)
            .addStreamTracerFactory(new MetricServerStreamTracer.Factory(metrics))
            .withChildOption(ChannelOption.ALLOCATOR, ByteBufAllocator.DEFAULT)
            .withChildOption(ChannelOption.TCP_NODELAY, true)
            .withChildOption(ChannelOption.SO_SNDBUF, 10 << 20) // 10 MiB
            .withChildOption(ChannelOption.SO_RCVBUF, 10 << 20) // 10 MiB
            .permitKeepAliveTime(1, TimeUnit.MINUTES)
            .permitKeepAliveWithoutCalls(true)
            .flowControlWindow(10 << 20); // 10 MiB

        builder.intercept(TracingServerInterceptor.newBuilder().build());

        setTime(builder::keepAliveTime, config.getKeepAliveTime());
        setTime(builder::keepAliveTimeout, config.getKeepAliveTimeout());

        for (int i = 1; i < config.getPortCount(); i++) {
            builder.addListenAddress(new InetSocketAddress(config.getPort(i)));
        }

        setInt(builder::maxInboundMessageSize, DataSizeConverter.toBytesInt(config.getMaxMessageSize()));

        for (BindableService service : services) {
            builder.addService(service);
        }
        if (!config.getDisableReflection()) {
            builder.addService(ProtoReflectionService.newInstance());
        }

        builder.channelType(NettyUtils.serverChannelType());
        builder.workerEventLoopGroup(threadpool.getIOExecutor());
        builder.bossEventLoopGroup(threadpool.getIOExecutor());
        return builder;
    }
}
