package ru.yandex.grpc.utils;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.config.protobuf.rpc.LimiterConfig;

import static ru.yandex.solomon.config.OptionalSet.setTime;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class LimiterOptions {
    public final boolean disable;
    public final long minRttNanos;
    public final int minLimit;
    public final int maxLimit;
    public final double rttTolerance;
    public final boolean useAsDefault;
    public final Set<String> endpoints;

    private LimiterOptions(Builder builder) {
        this.disable = builder.disable;
        this.minRttNanos = builder.minRttNanos;
        this.minLimit = builder.minLimit;
        this.maxLimit = builder.maxLimit;
        this.rttTolerance = builder.rttTolerance;
        this.useAsDefault = builder.useAsDefault;
        this.endpoints = builder.endpoints;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    @ParametersAreNonnullByDefault
    public static final class Builder {
        private boolean disable;
        private long minRttNanos;
        private int minLimit;
        private int maxLimit;
        private double rttTolerance;
        private boolean useAsDefault;
        private final Set<String> endpoints = new HashSet<>();

        public Builder setDisable(boolean disable) {
            this.disable = disable;
            return this;
        }

        public Builder setMinRttNanos(long minRttNanos) {
            this.minRttNanos = minRttNanos;
            return this;
        }

        public Builder setMinRttNanos(long value, TimeUnit unit) {
            this.minRttNanos = unit.toNanos(value);
            return this;
        }

        public Builder setMinLimit(int minLimit) {
            this.minLimit = minLimit;
            return this;
        }

        public Builder setMaxLimit(int maxLimit) {
            this.maxLimit = maxLimit;
            return this;
        }

        public Builder setRttTolerance(double rttTolerance) {
            this.rttTolerance = rttTolerance;
            return this;
        }

        public Builder setUseAsDefault(boolean useAsDefault) {
            this.useAsDefault = useAsDefault;
            return this;
        }

        public Builder addEndpoints(Collection<String> endpoints) {
            this.endpoints.addAll(endpoints);
            return this;
        }

        public Builder addEndpoint(String endpoint) {
            this.endpoints.add(endpoint);
            return this;
        }

        public Builder setFromConfig(LimiterConfig config) {
            setDisable(config.getDisable());
            setTime(this::setMinRttNanos, config.getMinRtt());
            setMinLimit(config.getMinLimit());
            setMaxLimit(config.getMaxLimit());
            setRttTolerance(config.getRttTolerance());
            setUseAsDefault(config.getUseAsDefault());
            addEndpoints(config.getEndpointsList());
            return this;
        }

        public LimiterOptions build() {
            return new LimiterOptions(this);
        }
    }
}
