package ru.yandex.grpc.utils;

import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.net.HostAndPort;
import io.grpc.ManagedChannel;
import io.grpc.netty.NegotiationType;
import io.grpc.netty.NettyChannelBuilder;
import io.netty.buffer.ByteBufAllocator;
import io.netty.channel.ChannelOption;
import io.opentracing.contrib.grpc.ActiveSpanContextSource;
import io.opentracing.contrib.grpc.ActiveSpanSource;
import io.opentracing.contrib.grpc.TracingClientInterceptor;

import ru.yandex.grpc.utils.client.interceptors.AddTokenInterceptor;
import ru.yandex.grpc.utils.client.interceptors.LimiterClientInterceptor;
import ru.yandex.grpc.utils.client.interceptors.LimiterProviderImpl;
import ru.yandex.grpc.utils.client.interceptors.MetricClientInterceptor;
import ru.yandex.solomon.util.NettyUtils;

import static ru.yandex.solomon.config.OptionalSet.setInt;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class NettyChannelFactory implements ChannelFactory {
    public static final int DEFAULT_CONNECT_TIMEOUT = Math.toIntExact(TimeUnit.SECONDS.toMillis(10));

    @Override
    public ManagedChannel createChannel(HostAndPort address, GrpcClientOptions options) {
        String host = address.getHost();
        int port = address.getPort();

        // Create InetSocketAddress to avoid use delayed dns resolve on each call
        // that use CachedThreadPool see https://github.com/grpc/grpc-java/issues/3703
        NettyChannelBuilder channelBuilder = NettyChannelBuilder.forAddress(host, port)
                .negotiationType(options.isUseTls() ? NegotiationType.TLS : NegotiationType.PLAINTEXT)
                .withOption(ChannelOption.ALLOCATOR, ByteBufAllocator.DEFAULT)
                .withOption(ChannelOption.TCP_NODELAY, Boolean.TRUE)
                .withOption(ChannelOption.CONNECT_TIMEOUT_MILLIS, DEFAULT_CONNECT_TIMEOUT)
                .withOption(ChannelOption.SO_SNDBUF, 10 << 20) // 10 MiB
                .withOption(ChannelOption.SO_RCVBUF, 10 << 20) // 10 MiB
                .flowControlWindow(10 << 20) // 10 MiB
                .maxInboundMetadataSize(10 << 20); // 10 MiB

        if (!options.getLimiterOptions().isEmpty()) {
            var limiterProvider = new LimiterProviderImpl(options.getLimiterOptions(), options.getMetricRegistry());
            channelBuilder.intercept(new LimiterClientInterceptor(limiterProvider));
        }

        channelBuilder.intercept(new MetricClientInterceptor(options.getClientId(), options.getMetricRegistry()));
        channelBuilder.intercept(TracingClientInterceptor.newBuilder()
                .withActiveSpanSource(ActiveSpanSource.GRPC_CONTEXT)
                .withActiveSpanContextSource(ActiveSpanContextSource.GRPC_CONTEXT)
                .build());
        if (options.getCallCredentialsSupplier().isPresent()) {
            channelBuilder.intercept(new AddTokenInterceptor(options.getCallCredentialsSupplier().get()));
        }
        setInt(channelBuilder::maxInboundMessageSize, options.getMaxInboundMessageSizeInBytes());

        if (options.getIdleTimeoutMillis() > 0) {
            channelBuilder.idleTimeout(options.getIdleTimeoutMillis(), TimeUnit.MILLISECONDS);
        }

        if (options.getKeepAliveDelayMillis() > 0) {
            channelBuilder.keepAliveTime(options.getKeepAliveDelayMillis(), TimeUnit.MILLISECONDS);
            channelBuilder.keepAliveTimeout(options.getKeepAliveTimeoutMillis(), TimeUnit.MILLISECONDS);
            channelBuilder.keepAliveWithoutCalls(true);
        }

        //TODO: don't use deprecated gRPC API <SOLOMON-8538>
        options.getNameResolverFactory().ifPresent(channelBuilder::nameResolverFactory);
        options.getRpcExecutor().ifPresent(channelBuilder::executor);
        options.getRpcExecutor().ifPresent(channelBuilder::offloadExecutor);
        options.getEventLoopGroup().ifPresent(eventLoop -> {
            channelBuilder.eventLoopGroup(eventLoop);
            channelBuilder.channelType(NettyUtils.clientChannelTypeForEventLoop(eventLoop));
        });
        return channelBuilder.build();
    }
}
