package ru.yandex.grpc.utils;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URI;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Attributes;
import io.grpc.EquivalentAddressGroup;
import io.grpc.NameResolver;
import io.grpc.Status;
import io.netty.resolver.dns.DnsNameResolver;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.actors.PingActorRunner;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.util.concurrent.MoreExecutors.directExecutor;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;
import static ru.yandex.solomon.util.NettyUtils.toCompletableFuture;


/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class NettyDnsNameResolver extends NameResolver {
    private static final String SCHEME = "dns";

    private final String host;
    private final int port;
    private final String authority;
    private final DnsNameResolver dns;
    private final AsyncMetrics metrics;
    private final PingActorRunner actor;

    @Nullable
    private volatile Listener listener;

    private NettyDnsNameResolver(
        String host,
        int port,
        String authority,
        DnsNameResolver dns,
        AsyncMetrics metrics,
        ScheduledExecutorService timer)
    {
        this.host = host;
        this.port = port;
        this.authority = authority;
        this.dns = dns;
        this.metrics = metrics;
        this.actor = PingActorRunner.newBuilder()
            .pingInterval(Duration.ofMinutes(5))
            .backoffDelay(Duration.ofMinutes(1))
            .backoffMaxDelay(Duration.ofMinutes(10))
            .operation("dns resolve of " + authority)
            .executor(directExecutor())
            .timer(timer)
            .onPing(this::act)
            .build();
    }

    public static Factory newFactory(DnsNameResolver resolver, ScheduledExecutorService timer, MetricRegistry registry) {
        return new DnsFactory(resolver, timer, registry);
    }

    @Override
    public String getServiceAuthority() {
        return authority;
    }

    @Override
    public void start(Listener listener) {
        this.listener = listener;
        actor.forcePing();
    }

    @Override
    public void refresh() {
        actor.forcePing();
    }

    @Override
    public void shutdown() {
        actor.close();
    }

    private CompletableFuture<?> act(int attempt) {
        var listener = this.listener;
        if (listener != null) {
            return resolveAll(listener);
        }

        return completedFuture(null);
    }

    private CompletableFuture<?> resolveAll(Listener listener) {
        var future = resolveAll();
        metrics.forFuture(future);
        return future.whenComplete((addresses, thr) -> {
            if (thr != null) {
                listener.onError(Status.INTERNAL.withCause(thr).withDescription("cannot resolve host: " + host));
            } else {
                listener.onAddresses(addressGroup(addresses), Attributes.EMPTY);
            }
        });
    }

    private CompletableFuture<List<InetAddress>> resolveAll() {
        return safeCall(() -> toCompletableFuture(dns.resolveAll(host)));
    }

    private List<EquivalentAddressGroup> addressGroup(List<InetAddress> addresses) {
        List<EquivalentAddressGroup> result = new ArrayList<>(addresses.size());
        for (var address : addresses) {
            result.add(new EquivalentAddressGroup(new InetSocketAddress(address, port)));
        }
        return result;
    }

    /**
     * FACTORY
     */
    @ParametersAreNonnullByDefault
    private static final class DnsFactory extends Factory {
        private final DnsNameResolver resolver;
        private final ScheduledExecutorService timer;
        private final AsyncMetrics metrics;

        DnsFactory(
            DnsNameResolver resolver,
            ScheduledExecutorService timer,
            MetricRegistry registry)
        {
            this.resolver = resolver;
            this.timer = timer;
            this.metrics = new AsyncMetrics(registry, "netty.dns.resolve");
        }

        @Nullable
        @Override
        public NameResolver newNameResolver(URI targetUri, Args args) {
            if (!SCHEME.equals(targetUri.getScheme())) {
                return null;
            }

            var targetPath = checkNotNull(targetUri.getPath(), "targetPath");
            checkArgument(
                targetPath.startsWith("/"),
                "the path component (%s) of the target (%s) must start with '/'", targetPath, targetUri);
            var name = targetPath.substring(1);

            // Must prepend a "//" to the name when constructing a URI, otherwise it will be treated as an
            // opaque URI, thus the authority and host of the resulted URI would be null.
            var nameUri = URI.create("//" + name);
            checkArgument(nameUri.getHost() != null, "Invalid DNS name: %s", name);

            var host = nameUri.getHost();
            int port = nameUri.getPort() != -1
                ? nameUri.getPort()
                : args.getDefaultPort();

            var authority = checkNotNull(
                nameUri.getAuthority(),
                "nameUri (%s) doesn't have an authority", nameUri);

            return new NettyDnsNameResolver(
                host,
                port,
                authority,
                resolver,
                metrics,
                timer);
        }

        @Override
        public String getDefaultScheme() {
            return SCHEME;
        }
    }
}
