package ru.yandex.grpc.utils;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Context;
import io.grpc.Contexts;
import io.grpc.Metadata;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;

import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class RequestParamsInterceptor implements ServerInterceptor {
    private static final Context.Key<String> REQUEST_ID_KEY = Context.key("XRequestId");
    private static final Context.Key<String> METHOD_NAME_KEY = Context.key("MethodName");
    private static final Context.Key<String> USER_IP_KEY = Context.key("UserIp");

    private static final Metadata.Key<String> REQUEST_ID =
            Metadata.Key.of("x-request-id", Metadata.ASCII_STRING_MARSHALLER);

    @Override
    public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(
            ServerCall<ReqT, RespT> call,
            Metadata headers,
            ServerCallHandler<ReqT, RespT> next)
    {
        String methodName = call.getMethodDescriptor().getFullMethodName();
        String requestId = Nullables.orEmpty(headers.get(REQUEST_ID));
        String userIp = GrpcUtils.realOrRemoteIp(call.getAttributes());

        Context context = Context.current().withValues(
                METHOD_NAME_KEY, methodName,
                REQUEST_ID_KEY, requestId,
                USER_IP_KEY, userIp);

        return Contexts.interceptCall(context, call, headers, next);
    }

    public static String getMethodName(Context context) {
        return METHOD_NAME_KEY.get(context);
    }

    public static String getRequestId(Context context) {
        return REQUEST_ID_KEY.get(context);
    }

    public static String getUserIp(Context context) {
        return USER_IP_KEY.get(context);
    }
}
