package ru.yandex.grpc.utils;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Flow;
import java.util.concurrent.Flow.Subscriber;
import java.util.concurrent.Flow.Subscription;
import java.util.function.Function;

import io.grpc.stub.StreamObserver;

/**
 * @author Vladimir Gordiychuk
 */
public final class StreamObservers {
    private StreamObservers() {
    }

    public static <T> void asyncComplete(CompletableFuture<T> future, StreamObserver<T> observer) {
        future.whenComplete((response, throwable) -> {
            if (throwable != null) {
                observer.onError(throwable);
            } else {
                observer.onNext(response);
                observer.onCompleted();
            }
        });
    }

    public static <Out, In> StreamObserver<In> map(StreamObserver<Out> observer, Function<? super In, ? extends Out> mapper) {
        return new StreamObserver<In>() {
            @Override
            public void onNext(In value) {
                Out mapped = mapper.apply(value);
                observer.onNext(mapped);
            }

            @Override
            public void onError(Throwable t) {
                observer.onError(t);
            }

            @Override
            public void onCompleted() {
                observer.onCompleted();
            }
        };
    }

    public static <Out> Flow.Subscriber<Out> toFlowSubscriber(StreamObserver<Out> out) {
        return new Subscriber<>() {
            @Override
            public void onSubscribe(Subscription subscription) {
                subscription.request(Long.MAX_VALUE);
            }

            @Override
            public void onNext(Out item) {
                out.onNext(item);
            }

            @Override
            public void onError(Throwable throwable) {
                out.onError(throwable);
            }

            @Override
            public void onComplete() {
                out.onCompleted();
            }
        };
    }

    public static <T> StreamObserver<T> noop() {
        return new NoopStreamObserver();
    }

    private static class NoopStreamObserver<T> implements StreamObserver<T> {

        @Override
        public void onNext(Object value) {
        }

        @Override
        public void onError(Throwable t) {
        }

        @Override
        public void onCompleted() {
        }
    }
}
