package ru.yandex.grpc.utils.client;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import io.grpc.Status;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
public class EndpointMetrics {
    private final MetricRegistry registry;
    private final ConcurrentMap<String, EndpointMetric> endpoints = new ConcurrentHashMap<>();

    public EndpointMetrics(MetricRegistry registry) {
        this.registry = registry;
    }

    public EndpointMetric byEndpoint(String endpoint) {
        var metrics = endpoints.get(endpoint);
        if (metrics != null) {
            return metrics;
        }

        return endpoints.computeIfAbsent(endpoint, name -> new EndpointMetric(name, registry));
    }

    public static class EndpointMetric {
        private final MetricRegistry registry;
        private final Rate inboundMessage;
        private final Rate outboundMessage;
        private final Rate inboundBytes;
        private final Rate outboundBytes;
        private final Rate callStarted;
        private final Rate callCompleted;
        private final LazyGaugeInt64 callInFlight;
        private final Histogram responseTime;
        private final Histogram inboundDeliveryTime;

        private final ConcurrentMap<Status.Code, Rate> callStatus = new ConcurrentHashMap<>();

        public EndpointMetric(String endpoint, MetricRegistry registry) {
            this(registry.subRegistry("endpoint", endpoint));
        }

        public EndpointMetric(MetricRegistry registry) {
            this.registry = registry;
            inboundMessage = registry.rate("grpc.client.call.inBoundMessages");
            outboundMessage = registry.rate("grpc.client.call.outBoundMessages");
            inboundBytes = registry.rate("grpc.client.call.inBoundBytes");
            outboundBytes = registry.rate("grpc.client.call.outBoundBytes");
            callStarted = registry.rate("grpc.client.call.started");
            callCompleted = registry.rate("grpc.client.call.completed");
            callInFlight = registry.lazyGaugeInt64("grpc.client.call.inFlight",
                    () -> callStarted.get() - callCompleted.get());
            responseTime = this.registry.histogramRate("grpc.client.call.elapsedTimeMs",
                    Histograms.exponential(20, 2, 1));
            inboundDeliveryTime = this.registry.histogramRate("grpc.client.call.delivery.elapsedTimeMs",
                    Histograms.exponential(16, 2, 1));
        }

        public void callStarted() {
            callStarted.inc();
        }

        private Rate callStatus(Status.Code status) {
            var result = callStatus.get(status);
            if (result != null) {
                return result;
            }

            result = registry.rate("grpc.client.call.status", Labels.of("code", status.name()));
            callStatus.putIfAbsent(status, result);
            return result;
        }

        public void callCompleted(Status status, long elapsedTime) {
            callCompleted.inc();
            callStatus(status.getCode()).inc();
            responseTime.record(elapsedTime);
        }

        public void addOutboundMessage() {
            this.outboundMessage.inc();
        }

        public void addInboundMessage() {
            this.inboundMessage.inc();
        }

        public void addOutboundBytes(long bytes) {
            this.outboundBytes.add(bytes);
        }

        public void addInboundBytes(long bytes) {
            this.inboundBytes.add(bytes);
        }

        public void firstMessageReceived(long elapsedTimeMs) {
            inboundDeliveryTime.record(elapsedTimeMs);
        }
    }
}
