package ru.yandex.grpc.utils.client.interceptors;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.Nullable;

import io.grpc.CallOptions;
import io.grpc.MethodDescriptor;

import ru.yandex.concurrency.limits.actors.Limiter;
import ru.yandex.concurrency.limits.actors.LimiterImpl;
import ru.yandex.concurrency.limits.actors.LimiterNoop;
import ru.yandex.grpc.utils.LimiterOptions;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.OptionalSet;

/**
 * @author Vladimir Gordiychuk
 */
public class LimiterProviderImpl implements LimiterProvider {
    private final MetricRegistry registry;
    private final List<LimiterOptions> options;
    private final Map<String, Limiter> limiterByEndpoint = new ConcurrentHashMap<>();

    public LimiterProviderImpl(List<LimiterOptions> options, MetricRegistry registry) {
        this.options = options;
        this.registry = registry;
    }

    @Override
    public <ReqT, RespT> Limiter provide(MethodDescriptor<ReqT, RespT> method, CallOptions callOptions) {
        var limiter = limiterByEndpoint.get(method.getFullMethodName());
        if (limiter != null) {
            return limiter;
        }

        return limiterByEndpoint.computeIfAbsent(method.getFullMethodName(), this::create);
    }

    private Limiter create(String endpoint) {
        var opts = findOptions(endpoint);
        if (opts == null || opts.disable) {
            return LimiterNoop.INSTANCE;
        }

        var builder = LimiterImpl.newBuilder();
        OptionalSet.setInt(builder::minLimit, opts.minLimit);
        OptionalSet.setInt(builder::maxLimit, opts.maxLimit);
        OptionalSet.setInt(builder::initLimit, opts.minLimit);
        OptionalSet.setLong(builder::minRtt, opts.minRttNanos);
        OptionalSet.setDouble(builder::rttTolerance, opts.rttTolerance);
        builder.registry(registry);
        builder.operation("grpc.client.call/" + endpoint);
        return builder.build();
    }

    @Nullable
    private LimiterOptions findOptions(String endpoint) {
        for (var opts : options) {
            if (opts.endpoints.contains(endpoint)) {
                return opts;
            }
        }

        for (var opts : options) {
            if (opts.useAsDefault) {
                return opts;
            }
        }

        return null;
    }
}
