package ru.yandex.grpc.utils.server;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;

/**
 * @author Vladimir Gordiychuk
 */
public class ServerMetrics implements MetricSupplier {
    private final ConcurrentMap<String, ClientMetrics> clients = new ConcurrentHashMap<>();

    public EndpointMetrics getEndpoint(String client, String fullMethodName) {
        return clients.computeIfAbsent(client, ClientMetrics::new).getEndpoint(fullMethodName);
    }

    @Override
    public int estimateCount() {
        return clients.values()
                .stream()
                .mapToInt(ClientMetrics::estimateCount)
                .sum();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        ClientMetrics total = new ClientMetrics("total");
        for (ClientMetrics client : clients.values()) {
            client.append(tsMillis, commonLabels, consumer);
            total.combine(client);
        }
        total.append(tsMillis, commonLabels, consumer);
    }

    private static class ClientMetrics implements MetricSupplier {
        private final String clientId;
        private final ConcurrentMap<String, EndpointMetrics> endpoints = new ConcurrentHashMap<>();

        public ClientMetrics(String client) {
            this.clientId = client;
        }

        public EndpointMetrics getEndpoint(String fullMethodName) {
            return endpoints.computeIfAbsent(fullMethodName, EndpointMetrics::new);
        }

        @Override
        public int estimateCount() {
            return endpoints.values()
                    .stream()
                    .mapToInt(EndpointMetrics::estimateCount)
                    .sum();
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            commonLabels = commonLabels.add("clientId", clientId);
            for (EndpointMetrics endpoint : endpoints.values()) {
                endpoint.append(tsMillis, commonLabels, consumer);
            }
        }

        public void combine(ClientMetrics target) {
            for (Map.Entry<String, EndpointMetrics> entry : target.endpoints.entrySet()) {
                getEndpoint(entry.getKey()).combine(entry.getValue());
            }
        }
    }
}
