package ru.yandex.grpc.utils.server.interceptors;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import io.grpc.Metadata;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import io.grpc.Status;

import ru.yandex.grpc.utils.Headers;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;


/**
 * @author Vladimir Gordiychuk
 */
@LinkedOnRootPage("gRPC clients filter")
public class ClientFilterServerInterceptor implements ServerInterceptor {
    private final ConcurrentMap<String, ClientStatus> clients = new ConcurrentHashMap<>();

    @Override
    public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(ServerCall<ReqT, RespT> call, Metadata headers, ServerCallHandler<ReqT, RespT> next) {
        String client = Headers.getClient(headers);
        ClientStatus status = clients.getOrDefault(client, ClientStatus.ENABLE);
        if (status == ClientStatus.ENABLE) {
            return next.startCall(call, headers);
        }

        call.close(Status.PERMISSION_DENIED.withDescription("client " + client + " disabled"), headers);
        return Noop.getInstance();
    }

    @ManagerMethod
    public void enableAll() {
        clients.clear();
    }

    @ManagerMethod
    public void enableClient(String client) {
        clients.put(client, ClientStatus.ENABLE);
    }

    @ManagerMethod
    public void disableClient(String client) {
        clients.put(client, ClientStatus.DISABLE);
    }

    private enum ClientStatus {
        ENABLE, DISABLE
    }

    private static class Noop<T> extends ServerCall.Listener<T> {
        private static final Noop INSTANCE = new Noop();

        @SuppressWarnings("unchecked")
        static <T> ServerCall.Listener<T> getInstance() {
            return INSTANCE;
        }
    }
}
