package ru.yandex.grpc.utils.client.interceptors;

import java.io.StringWriter;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import io.grpc.CallOptions;
import io.grpc.ManagedChannel;
import io.grpc.Status;
import io.grpc.Status.Code;
import io.grpc.health.v1.HealthCheckRequest;
import io.grpc.health.v1.HealthCheckResponse;
import io.grpc.health.v1.HealthGrpc;
import io.grpc.inprocess.InProcessChannelBuilder;
import io.grpc.testing.GrpcServerRulePublic;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.grpc.utils.SingleResponseStreamObserver;
import ru.yandex.monlib.metrics.encode.text.MetricTextEncoder;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static io.grpc.stub.ClientCalls.asyncUnaryCall;
import static org.hamcrest.Matchers.containsString;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricClientInterceptorTest {

    @Rule
    public GrpcServerRulePublic server = new GrpcServerRulePublic();
    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(30, TimeUnit.SECONDS)
            .build();

    private HealthServiceStub service;
    private ManagedChannel channel;
    private MetricRegistry registry;

    @Before
    public void setUp() throws Exception {
        service = new HealthServiceStub();
        server.getServiceRegistry().addService(service);
        registry = new MetricRegistry();
        channel = InProcessChannelBuilder.forName(server.getServerName())
                .usePlaintext()
                .intercept(new MetricClientInterceptor("test", registry))
                .build();
    }

    @After
    public void tearDown() throws Exception {
        if (channel != null) {
            channel.shutdownNow();
        }
    }

    @Test
    public void callInflight() throws InterruptedException {
        var futureOne = healthCheck();
        expectInflight(1);

        var futureTwo = healthCheck();
        expectInflight(2);

        service.calls.take().complete();
        CompletableFuture.anyOf(futureOne, futureTwo).join();
        expectInflight(1);

        service.calls.take().complete();
        CompletableFuture.allOf(futureOne, futureTwo).join();
        expectInflight(0);
        expectStatus(Code.OK, 2);
    }

    @Test
    public void countStatusWhenDeadlineEarlier() {
        var status = healthCheck(CallOptions.DEFAULT.withDeadlineAfter(1, TimeUnit.NANOSECONDS)).join();
        assertEquals(status.toString(), Status.Code.DEADLINE_EXCEEDED, status.getCode());
        expectStatus(Code.DEADLINE_EXCEEDED, 1);
    }

    private void expectInflight(int count) {
        assertThat(metrics(), containsString("IGAUGE grpc.client.call.inFlight{endpoint='grpc.health.v1.Health/Check'} [" + count + "]"));
    }

    private void expectStatus(Status.Code code, int count) {
        String expect = "RATE grpc.client.call.status{code='" + code.name() + "', endpoint='grpc.health.v1.Health/Check'} [" + count + "]";
        assertThat(metrics(), containsString(expect));
    }

    private String metrics() {
        StringWriter writer = new StringWriter();
        try (MetricTextEncoder e = new MetricTextEncoder(writer, true)) {
            registry.supply(0, e);
        }
        return writer.toString();
    }

    private CompletableFuture<Status> healthCheck() {
        return healthCheck(CallOptions.DEFAULT);
    }

    private CompletableFuture<Status> healthCheck(CallOptions callOptions) {
        assertNotNull(channel);
        var future = new CompletableFuture<HealthCheckResponse>();
        var call = channel.newCall(HealthGrpc.getCheckMethod(), callOptions);
        asyncUnaryCall(call, HealthCheckRequest.getDefaultInstance(), new SingleResponseStreamObserver<>(future));
        return future.thenApply(ignore -> Status.OK).exceptionally(Status::fromThrowable);
    }
}
