package ru.yandex.solomon.http.filters;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.stereotype.Component;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebFilter;
import org.springframework.web.server.WebFilterChain;
import reactor.core.publisher.Mono;

import ru.yandex.solomon.util.http.HttpUtils;

/**
 * @author Stepan Koltsov
 */
@Component
@ParametersAreNonnullByDefault
public class RequestLoggingFilter implements WebFilter {
    private static final Logger logger = LoggerFactory.getLogger("access");

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, WebFilterChain chain) {
        ExchangeLog entry = new ExchangeLog(exchange);
        long startMillis = System.currentTimeMillis();
        logger.info(entry.format() + "...");

        return chain.filter(exchange)
                .doFinally((signalType) -> {
                    long elapsedMillis = System.currentTimeMillis() - startMillis;
                    var statusCode = exchange.getResponse().getRawStatusCode();
                    logger.info(entry.format() + " " + statusCode + " " + elapsedMillis + " ms");
                });
    }

    private static class ExchangeLog {
        private final ServerWebExchange exchange;
        private final String remoteAddr;
        private final String methodAndUri;
        private final String requestId;

        public ExchangeLog(ServerWebExchange exchange) {
            this.exchange = exchange;
            ServerHttpRequest request = exchange.getRequest();
            this.remoteAddr = HttpUtils.realOrRemoteIp(request) + ' ';
            HttpMethod method = request.getMethod();
            String methodStr = method != null ? method.toString() : "UNKNOWN";
            String uri = HttpUtils.requestUri(request);
            this.methodAndUri = methodStr + ' ' + uri;
            this.requestId = HttpUtils.requestId(request);
        }

        public String format() {
            @Nullable String authSubject = exchange.getAttribute("ru.yandex.solomon.auth.AuthSubject");
            String maybeAuthSubject = authSubject != null ? authSubject + " " : "";
            return remoteAddr + maybeAuthSubject + methodAndUri + " requestId:" + requestId;
        }
    }
}
