package ru.yandex.solomon.idempotency.dao.ydb;

import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.transaction.TxControl;

import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.idempotency.IdempotentOperation;
import ru.yandex.solomon.idempotency.dao.IdempotentOperationDao;

import static com.yandex.ydb.table.values.PrimitiveValue.string;
import static com.yandex.ydb.table.values.PrimitiveValue.timestamp;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.COMPLETED_AT;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.CONTAINER_ID;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.CONTAINER_TYPE;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.ENTITY_ID;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.ID;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.OPERATION_TYPE;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.PARAM_COMPLETED_AT;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.PARAM_CONTAINER_ID;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.PARAM_CONTAINER_TYPE;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.PARAM_ENTITY_ID;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.PARAM_ID;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.PARAM_OPERATION_TYPE;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.PARAM_RESULT;
import static ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationTable.RESULT;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class YdbIdempotentOperationDao implements IdempotentOperationDao {

    private final SessionRetryContext retryCtx;
    private final SchemeClient schemeClient;
    private final YdbIdempotentOperationQuery query;

    public YdbIdempotentOperationDao(String root, TableClient tableClient, SchemeClient schemeClient) {
        this.schemeClient = schemeClient;
        this.query = new YdbIdempotentOperationQuery(root);
        this.retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .sessionSupplyTimeout(Duration.ofSeconds(30))
                .build();
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return schemeClient.makeDirectories(query.root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(unused -> createTable(query.table, YdbIdempotentOperationTable::createTable));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        Function<Session, CompletableFuture<Status>> doDrop = (session) -> session.dropTable(query.table);
        return retryCtx.supplyStatus(doDrop)
                .thenAccept(status -> status.expect("cannot drop " + query.table + " table"));
    }

    @Override
    public CompletableFuture<Boolean> complete(IdempotentOperation operation) {
        var params = Params.create()
                .put(PARAM_ID, utf8(operation.id()))
                .put(PARAM_CONTAINER_ID, utf8(operation.containerId()))
                .put(PARAM_CONTAINER_TYPE, utf8(operation.containerType().name()))
                .put(PARAM_OPERATION_TYPE, utf8(operation.operationType()))
                .put(PARAM_ENTITY_ID, utf8(operation.entityId()))
                .put(PARAM_RESULT, string(operation.result().toByteString()))
                .put(PARAM_COMPLETED_AT, timestamp(Instant.ofEpochMilli(operation.completedAt())));

        return execute(query.complete, params)
                .thenApply(result -> {
                    if (result.isSuccess()) {
                        return Boolean.TRUE;
                    }

                    if (result.getCode() == StatusCode.PRECONDITION_FAILED) {
                        return Boolean.FALSE;
                    }

                    result.expect("unable insert new operation");
                    return Boolean.FALSE;
                });
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(IdempotentOperation operation) {
        var params = Params.create()
                .put(PARAM_ID, utf8(operation.id()))
                .put(PARAM_CONTAINER_ID, utf8(operation.containerId()))
                .put(PARAM_CONTAINER_TYPE, utf8(operation.containerType().name()))
                .put(PARAM_OPERATION_TYPE, utf8(operation.operationType()));
        return execute(query.deleteOne, params)
                .thenApply(result -> {
                    var res = result.expect("cannot execute query");
                    ResultSetReader resultSet = res.getResultSet(0);
                    if (resultSet.next()) {
                        return resultSet.getColumn(0).getBool();
                    }
                    return Boolean.FALSE;
                });
    }

    @Override
    public CompletableFuture<Optional<IdempotentOperation>> get(String id, String containerId, ContainerType containerType, String operationType) {
        Params params = Params.of(
                PARAM_ID, utf8(id),
                PARAM_CONTAINER_ID, utf8(containerId),
                PARAM_CONTAINER_TYPE, utf8(containerType.name()),
                PARAM_OPERATION_TYPE, utf8(operationType)
        );
        return execute(query.selectOne, params)
                .thenApply(result -> {
                    var dataQueryResult = result.expect("unable select operation by id");
                    var rs = dataQueryResult.getResultSet(0);
                    if (!rs.next()) {
                        return Optional.empty();
                    }

                    return Optional.of(operation(rs));
                });
    }

    static Any any(ResultSetReader rs, String columnName) {
        try {
            return Any.parseFrom(rs.getColumn(columnName).getString());
        } catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Unable to parse column " + columnName + " at row " + rs.getRowCount());
        }
    }

    private CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
        try {
            return retryCtx.supplyResult(s -> {
                var settings = new ExecuteDataQuerySettings().keepInQueryCache();
                var tx = TxControl.serializableRw();
                return s.executeDataQuery(query, tx, params, settings);
            });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Void> createTable(String tablePath, BiFunction<String, Session, CompletableFuture<Status>> fn) {
        return schemeClient.describePath(tablePath)
                .thenCompose(exist -> {
                    if (exist.isSuccess()) {
                        return completedFuture(com.yandex.ydb.core.Status.SUCCESS);
                    }

                    return retryCtx.supplyStatus(session -> fn.apply(tablePath, session));
                })
                .thenAccept(status -> status.expect("cannot create operation table " + tablePath));
    }


    public static void fillExternalParams(Params params, IdempotentOperation operation) {
        params.put("$operationId", utf8(operation.id()))
                .put(PARAM_CONTAINER_ID, utf8(operation.containerId()))
                .put(PARAM_CONTAINER_TYPE, utf8(operation.containerType().name()))
                .put(PARAM_OPERATION_TYPE, utf8(operation.operationType()))
                .put(PARAM_ENTITY_ID, utf8(operation.entityId()))
                .put(PARAM_RESULT, string(operation.result().toByteString()))
                .put(PARAM_COMPLETED_AT, timestamp(Instant.ofEpochMilli(operation.completedAt())));
    }

    public static Params existParams(IdempotentOperation operation) {
        return Params.create()
                .put(PARAM_ID, utf8(operation.id()))
                .put(PARAM_CONTAINER_ID, utf8(operation.containerId()))
                .put(PARAM_CONTAINER_TYPE, utf8(operation.containerType().name()))
                .put(PARAM_OPERATION_TYPE, utf8(operation.operationType()));
    }

    public static IdempotentOperation operation(ResultSetReader rs) {
        return new IdempotentOperation(
                rs.getColumn(ID).getUtf8(),
                rs.getColumn(CONTAINER_ID).getUtf8(),
                ContainerType.valueOf(rs.getColumn(CONTAINER_TYPE).getUtf8()),
                rs.getColumn(OPERATION_TYPE).getUtf8(),
                rs.getColumn(ENTITY_ID).getUtf8(),
                any(rs, RESULT),
                rs.getColumn(COMPLETED_AT).getTimestamp().toEpochMilli()
        );
    }
}
