package ru.yandex.solomon.idempotency.dao;

import java.time.Clock;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Optional;
import java.util.UUID;

import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.StringValue;
import org.junit.Test;

import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.idempotency.IdempotentOperation;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Alexey Trushkin
 */
public abstract class AbstractIdempotentOperationDaoTest {
    private static final String JSON = "{\"a\":\"b\"}";
    private static final String ENTITY_ID = "someId";
    private static final String OP_TYPE = "veryImportantOp";
    private static final String CONTAINER_ID = "containerId";
    private final Clock constantClock = Clock.fixed(Instant.now(), ZoneId.systemDefault());

    public abstract IdempotentOperationDao getDao();

    @Test
    public void complete_existed() {
        String id = "id";
        assertTrue(completeSync(operation(id)));

        assertFalse(completeSync(operation(id)));
    }

    @Test
    public void get() {
        var operation = operation();
        assertTrue(completeSync(operation));

        var actual = findSync(operation.id());
        assertTrue(actual.isPresent());
        assertEquals(operation.result(), actual.get().result());
        assertEquals(operation.entityId(), actual.get().entityId());
        assertEquals(constantClock.millis(), actual.get().completedAt());
    }

    @Test
    public void complete() throws InvalidProtocolBufferException {
        var operation = new IdempotentOperation("id", CONTAINER_ID, ContainerType.PROJECT, OP_TYPE, ENTITY_ID, Any.pack(StringValue.of(JSON)), constantClock.millis());
        assertTrue(completeSync(operation));

        var actual = findSync(operation.id());
        assertTrue(actual.isPresent());
        assertEquals(ENTITY_ID, actual.get().entityId());
        assertEquals(JSON, actual.get().result().unpack(StringValue.class).getValue());
        assertEquals(constantClock.millis(), actual.get().completedAt());
    }

    @Test
    public void delete() throws InvalidProtocolBufferException {
        var operation = new IdempotentOperation("id", CONTAINER_ID, ContainerType.PROJECT, OP_TYPE, ENTITY_ID, Any.pack(StringValue.of(JSON)), constantClock.millis());
        assertTrue(completeSync(operation));
        findSync(operation.id());

        var result = delete(operation);
        assertTrue(result);
        assertTrue(findSync(operation.id()).isEmpty());

        result = delete(operation);
        assertFalse(result);

        result = delete(new IdempotentOperation("id1", CONTAINER_ID, ContainerType.PROJECT, OP_TYPE, ENTITY_ID, Any.pack(StringValue.of(JSON)), constantClock.millis()));
        assertFalse(result);
    }

    private boolean completeSync(IdempotentOperation operation) {
        return join(getDao().complete(operation));
    }

    private boolean delete(IdempotentOperation operation) {
        return join(getDao().deleteOne(operation));
    }

    private Optional<IdempotentOperation> findSync(String id) {
        return join(getDao().get(id, CONTAINER_ID, ContainerType.PROJECT, OP_TYPE));
    }

    private IdempotentOperation operation() {
        return operation(UUID.randomUUID().toString());
    }

    private IdempotentOperation operation(String id) {
        return new IdempotentOperation(id, CONTAINER_ID, ContainerType.PROJECT, OP_TYPE, ENTITY_ID, Any.getDefaultInstance(), constantClock.millis());
    }
}
