package ru.yandex.kikimr.client;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import io.netty.util.HashedWheelTimer;
import io.netty.util.Timeout;
import io.netty.util.Timer;
import io.netty.util.TimerTask;
import io.netty.util.concurrent.DefaultThreadFactory;

import ru.yandex.misc.concurrent.CompletableFutures;


/**
 * @author Sergey Polovko
 */
public final class KikimrAsyncRetry {
    private KikimrAsyncRetry() {
    }

    public static final class RetryConfig {
        private final int maxRetries;
        private final long sleepMinMillis;
        private final long sleepMaxMillis;

        private RetryConfig(int maxRetries, long sleepMinMillis, long sleepMaxMillis) {
            this.maxRetries = maxRetries;
            this.sleepMinMillis = sleepMinMillis;
            this.sleepMaxMillis = sleepMaxMillis;
        }

        public static RetryConfig maxRetries(int n) {
            return new RetryConfig(n, 100, 5_000);
        }

        public static RetryConfig maxRetriesAndTime(int maxRetries, long sleepMaxMillis) {
            return new RetryConfig(maxRetries, 100, sleepMaxMillis);
        }

        public static RetryConfig maxRetriesAndTime(int maxRetries, long sleepMinMillis, long sleepMaxMillis) {
            return new RetryConfig(maxRetries, sleepMinMillis, sleepMaxMillis);
        }

        public int getMaxRetries() {
            return maxRetries;
        }

        public long getSleepMinMillis() {
            return sleepMinMillis;
        }

        public long getSleepMaxMillis() {
            return sleepMaxMillis;
        }
    }

    private static final Timer timer = new HashedWheelTimer(
        new DefaultThreadFactory("KikimrAsyncRetryTimer", true),
        10, TimeUnit.MILLISECONDS);

    public static <T> CompletableFuture<T> withRetries(RetryConfig config, Supplier<CompletableFuture<T>> fn) {
        try {
            CompletableFuture<T> promise = new CompletableFuture<>();
            fn.get().whenComplete((r, t) -> {
                if (t == null) {
                    promise.complete(r);
                    return;
                }

                Throwable cause = CompletableFutures.unwrapCompletionException(t);
                if (!canRetry(cause)) {
                    promise.completeExceptionally(cause);
                    return;
                }

                AsyncRetryTask<T> task = new AsyncRetryTask<>(config, fn, promise);
                task.scheduleNext(timer);
            });

            return promise;
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    public static <T> CompletableFuture<T> withRetriesSync(RetryConfig config, Supplier<T> fn) {
        try {
            return CompletableFuture.completedFuture(fn.get());
        } catch (Throwable t) {
            CompletableFuture<T> promise = new CompletableFuture<>();
            if (canRetry(t)) {
                SyncRetryTask<T> task = new SyncRetryTask<>(config, fn, promise);
                task.scheduleNext(timer);
            } else {
                promise.completeExceptionally(t);
            }
            return promise;
        }
    }

    private static boolean canRetry(Throwable cause) {
        if (cause instanceof KikimrAnyResponseException) {
            return true;
        }
        if (cause instanceof UnableToConnectException) {
            return true;
        }
        if (cause instanceof RuntimeException) {
            String message = cause.getMessage();
            if (message != null && message.startsWith("unregistered;")) {
                return true;
            }
            if (message != null && message.contains("DEADLINE_EXCEEDED")) {
                return true;
            }
            return message != null && message.contains("RESOURCE_EXHAUSTED");
        }
        return false;
    }

    /**
     * ASYNC RETRY TASK
     */
    private static final class AsyncRetryTask<T> implements TimerTask {
        private final RetryConfig config;
        private final Supplier<CompletableFuture<T>> fn;
        private final CompletableFuture<T> promise;

        private int retryNum;
        private long delayMillis;

        AsyncRetryTask(
            RetryConfig config, Supplier<CompletableFuture<T>> fn, CompletableFuture<T> promise) {
            this.config = config;
            this.fn = fn;
            this.promise = promise;
            this.delayMillis = config.getSleepMinMillis();
        }

        @Override
        public void run(Timeout timeout) {
            try {
                fn.get().whenComplete((r, t) -> {
                    if (t == null) {
                        promise.complete(r);
                        return;
                    }

                    Throwable cause = CompletableFutures.unwrapCompletionException(t);
                    if (!canRetry(cause)) {
                        promise.completeExceptionally(cause);
                        return;
                    }

                    if (++retryNum < config.getMaxRetries()) {
                        scheduleNext(timeout.timer());
                    } else {
                        String msg = "async operation failed " + retryNum + " times, stop retrying it";
                        promise.completeExceptionally(new RuntimeException(msg, cause));
                    }
                });
            } catch (Throwable t) {
                promise.completeExceptionally(t);
            }
        }

        private void scheduleNext(Timer timer) {
            timer.newTimeout(this, delayMillis, TimeUnit.MILLISECONDS);
            // exponentially growing delay (1.5^i)
            delayMillis = Math.min(delayMillis + delayMillis / 2, config.getSleepMaxMillis());
        }
    }

    /**
     * SYNC RETRY TASK
     */
    private static final class SyncRetryTask<T> implements TimerTask {
        private final RetryConfig config;
        private final Supplier<T> fn;
        private final CompletableFuture<T> promise;

        private int retryNum;
        private long delayMillis;

        SyncRetryTask(RetryConfig config, Supplier<T> fn, CompletableFuture<T> promise) {
            this.config = config;
            this.fn = fn;
            this.promise = promise;
            this.delayMillis = config.getSleepMinMillis();
        }

        @Override
        public void run(Timeout timeout) {
            try {
                T value = fn.get();
                promise.complete(value);
            } catch (Throwable t) {
                if (!canRetry(t)) {
                    promise.completeExceptionally(t);
                } else if (++retryNum < config.getMaxRetries()) {
                    scheduleNext(timeout.timer());
                } else {
                    String msg = "operation failed " + retryNum + " times, stop retrying it";
                    promise.completeExceptionally(new RuntimeException(msg, t));
                }
            }
        }

        private void scheduleNext(Timer timer) {
            timer.newTimeout(this, delayMillis, TimeUnit.MILLISECONDS);
            // exponentially growing delay (1.5^i)
            delayMillis = Math.min(delayMillis + delayMillis / 2, config.getSleepMaxMillis());
        }
    }
}
