package ru.yandex.kikimr.client;

import java.time.Duration;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.WillNotClose;

import NKikimrClient.TGRpcServerGrpc;
import com.google.common.net.HostAndPort;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;

import ru.yandex.kikimr.client.discovery.Discovery;
import ru.yandex.kikimr.client.discovery.StaticDiscovery;
import ru.yandex.kikimr.client.kv.noderesolver.KikimrKvNodeResolver;
import ru.yandex.kikimr.client.kv.noderesolver.KikimrKvNodeResolverImpl;
import ru.yandex.kikimr.client.kv.noderesolver.KikimrKvNoopNodeResolver;
import ru.yandex.kikimr.grpc.GrpcOptions;
import ru.yandex.kikimr.grpc.GrpcTransport;
import ru.yandex.kikimr.proto.Msgbus;
import ru.yandex.kikimr.proto.MsgbusKv;


/**
 * @author Sergey Polovko
 */
public class KikimrGrpcTransport implements KikimrTransport {
    private final GrpcTransport impl;
    private final KikimrKvNodeResolver kvNodeResolver;

    public KikimrGrpcTransport(Collection<HostAndPort> hostAndPorts) {
        this(hostAndPorts, GrpcOptions.newBuilder().build());
    }

    public KikimrGrpcTransport(Collection<HostAndPort> hostAndPorts, int maxMessageSizeBytes) {
        this(hostAndPorts, GrpcOptions.newBuilder()
            .maxInboundMessageSize(maxMessageSizeBytes)
            .build());
    }

    public KikimrGrpcTransport(
        Collection<HostAndPort> hostAndPorts,
        int maxMessageSizeBytes,
        Duration connectTimeout,
        Duration readTimeout,
        @WillNotClose @Nullable EventLoopGroup ioExecutor,
        @WillNotClose @Nullable Executor callExecutor)
    {
        this(hostAndPorts, maxMessageSizeBytes, connectTimeout, readTimeout, ioExecutor, callExecutor, null);
    }

    public KikimrGrpcTransport(
        Collection<HostAndPort> hostAndPorts,
        int maxMessageSizeBytes,
        Duration connectTimeout,
        Duration readTimeout,
        @WillNotClose @Nullable EventLoopGroup ioExecutor,
        @WillNotClose @Nullable Executor callExecutor,
        @WillNotClose @Nullable ScheduledExecutorService timer)
    {
        this(hostAndPorts, GrpcOptions.newBuilder()
            .maxInboundMessageSize(maxMessageSizeBytes)
            .readTimeout(readTimeout.toMillis(), TimeUnit.MILLISECONDS)
            .withOption(ChannelOption.CONNECT_TIMEOUT_MILLIS, Math.toIntExact(connectTimeout.toMillis()))
            .eventLoopGroup(ioExecutor)
            .callExecutor(callExecutor)
            .timer(timer)
            .build());
    }

    public KikimrGrpcTransport(Collection<HostAndPort> hostAndPorts, GrpcOptions opts) {
        this(new StaticDiscovery(hostAndPorts), opts);
    }

    public KikimrGrpcTransport(Discovery discovery, GrpcOptions opts) {
        this.impl = new GrpcTransport(discovery, opts);
        if (opts.timer != null && opts.callExecutor != null) {
            kvNodeResolver = new KikimrKvNodeResolverImpl(this.impl, opts.callExecutor, opts.timer);
        } else {
            kvNodeResolver = new KikimrKvNoopNodeResolver();
        }
    }

    @Override
    public CompletableFuture<Msgbus.TResponse> schemeOperation(Msgbus.TSchemeOperation request) {
        return impl.unaryCall(TGRpcServerGrpc.getSchemeOperationMethod(), request, 0, null);
    }

    @Override
    public CompletableFuture<Msgbus.TResponse> schemeOperationStatus(Msgbus.TSchemeOperationStatus request) {
        return impl.unaryCall(TGRpcServerGrpc.getSchemeOperationStatusMethod(), request, 0, null);
    }

    @Override
    public CompletableFuture<Msgbus.TResponse> schemeDescribe(Msgbus.TSchemeDescribe request) {
        return impl.unaryCall(TGRpcServerGrpc.getSchemeDescribeMethod(), request, 0, null);
    }

    @Override
    public CompletableFuture<Msgbus.TLocalEnumerateTabletsResult> localEnumerateTablets(Msgbus.TLocalEnumerateTablets request) {
        return impl.unaryCall(TGRpcServerGrpc.getLocalEnumerateTabletsMethod(), request, 0, null)
            .thenApply(KikimrGrpcTransport::toLocalEnumerateTablets);
    }

    private static Msgbus.TLocalEnumerateTabletsResult toLocalEnumerateTablets(Msgbus.TResponse response) {
        Msgbus.TLocalEnumerateTabletsResult.Builder b = Msgbus.TLocalEnumerateTabletsResult.newBuilder()
            .setStatus(response.getStatus());
        if (response.hasErrorReason()) {
            b.setErrorReason(response.getErrorReason());
        }
        for (int i = 0; i < response.getTabletInfoCount(); i++) {
            Msgbus.TResponse.TTabletInfo tablet = response.getTabletInfo(i);
            b.addTabletInfoBuilder()
                .setTabletId(tablet.getTabletId())
                .setTabletType(tablet.getTabletType());
        }
        return b.build();
    }

    @Override
    public CompletableFuture<MsgbusKv.TKeyValueResponse> keyValue(MsgbusKv.TKeyValueRequest request) {
        var node = kvNodeResolver.resolveByTabletId(request.getTabletId());
        return impl.unaryCall(TGRpcServerGrpc.getKeyValueMethod(), request, request.getDeadlineInstantMs(), node)
            .thenApply(KikimrGrpcTransport::toKeyValue);
    }

    private static MsgbusKv.TKeyValueResponse toKeyValue(Msgbus.TResponse response) {
        MsgbusKv.TKeyValueResponse.Builder b = MsgbusKv.TKeyValueResponse.newBuilder()
            .setStatus(response.getStatus());
        if (response.hasCookie()) {
            b.setCookie(response.getCookie());
        }
        if (response.getDeleteRangeResultCount() > 0) {
            b.addAllDeleteRangeResult(response.getDeleteRangeResultList());
        }
        if (response.hasIncrementGenerationResult()) {
            b.setIncrementGenerationResult(response.getIncrementGenerationResult());
        }
        if (response.getReadRangeResultCount() > 0) {
            b.addAllReadRangeResult(response.getReadRangeResultList());
        }
        if (response.getReadResultCount() > 0) {
            b.addAllReadResult(response.getReadResultList());
        }
        if (response.getWriteResultCount() > 0) {
            b.addAllWriteResult(response.getWriteResultList());
        }
        if (response.getRenameResultCount() > 0) {
            b.addAllRenameResult(response.getRenameResultList());
        }
        if (response.getCopyRangeResultCount() > 0) {
            b.addAllCopyRangeResult(response.getCopyRangeResultList());
        }
        if (response.getConcatResultCount() > 0) {
            b.addAllConcatResult(response.getConcatResultList());
        }
        if (response.getGetStatusResultCount() > 0) {
            b.addAllGetStatusResult(response.getGetStatusResultList());
        }
        if (response.hasTrimLeakedBlobsResult()) {
            b.setTrimLeakedBlobsResult(response.getTrimLeakedBlobsResult());
        }
        if (response.hasSetExecutorFastLogPolicyResult()) {
            b.setSetExecutorFastLogPolicyResult(response.getSetExecutorFastLogPolicyResult());
        }
        if (response.hasErrorReason()) {
            b.setErrorReason(response.getErrorReason());
        }
        return b.build();
    }

    @Override
    public void close() {
        impl.close();
    }
}
