package ru.yandex.kikimr.client.discovery;

import java.time.Duration;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import javax.annotation.WillCloseWhenClosed;

import com.google.common.net.HostAndPort;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.grpc.GrpcDiscoveryRpc;
import com.yandex.ydb.core.grpc.GrpcRequestSettings;
import com.yandex.ydb.discovery.DiscoveryProtos.ListEndpointsResult;
import io.grpc.Status;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.solomon.util.actors.PingActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbDiscovery implements Discovery, NodeDiscovery, AutoCloseable {
    private final GrpcDiscoveryRpc rpc;
    private final String database;
    private final PingActorRunner actor;
    private volatile Int2ObjectOpenHashMap<HostAndPort> addresses = new Int2ObjectOpenHashMap<>();
    private final AtomicReference<CompletableFuture<Void>> onChange = new AtomicReference<>(new CompletableFuture<>());

    public YdbDiscovery(@WillCloseWhenClosed GrpcDiscoveryRpc rpc, String database, Executor executor,
            ScheduledExecutorService timer)
    {
        this.rpc = rpc;
        this.database = database;
        this.actor = PingActorRunner.newBuilder()
                .onPing(this::act)
                .executor(executor)
                .timer(timer)
                .operation("discovery_ydb_endpoints")
                .pingInterval(Duration.ofMinutes(1))
                .backoffMaxDelay(Duration.ofMinutes(5))
                .backoffDelay(Duration.ofSeconds(5))
                .build();
        actor.forcePing();
    }

    public Set<HostAndPort> addresses() {
        return Set.copyOf(addresses.values());
    }

    public Int2ObjectOpenHashMap<HostAndPort> nodeIdToAddresses() {
        return addresses;
    }

    public CompletionStage<Void> forceUpdate() {
        return actor.forcePing();
    }

    public CompletionStage<Void> onChange() {
        return onChange.get();
    }

    private CompletableFuture<Void> act(int attempt) {
        return rpc.listEndpoints(database, GrpcRequestSettings.newBuilder().build())
                .thenApply(result -> {
                    var response = ensureSuccessResolved(result);
                    var addresses = response.getEndpointsList()
                            .stream()
                            .map(endpointInfo -> Map.entry(endpointInfo.getNodeId(),
                                    HostAndPort.fromParts(endpointInfo.getAddress(), endpointInfo.getPort())))
                            .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (a, b) -> a));
                    if (!addresses.equals(this.addresses)) {
                        this.addresses = new Int2ObjectOpenHashMap<>(addresses);
                        onChange.getAndSet(new CompletableFuture<>()).complete(null);
                    }
                    return null;
                });
    }

    private ListEndpointsResult ensureSuccessResolved(Result<ListEndpointsResult> result) {
        if (!result.isSuccess()) {
            String msg = "unable to resolve database " + database +
                    ", got non SUCCESS response: " + result.getCode() +
                    ", issues: " + Arrays.toString(result.getIssues());
            throw Status.UNAVAILABLE.withDescription(msg).asRuntimeException();
        }

        ListEndpointsResult response = result.expect("listEndpoints()");
        if (response.getEndpointsCount() == 0) {
            String msg = "unable to resolve database " + database + ", got empty list of endpoints";
            throw Status.UNAVAILABLE.withDescription(msg).asRuntimeException();
        }

        return response;
    }

    @Override
    public void close() throws Exception {
        actor.close();
        rpc.close();
    }
}
