package ru.yandex.kikimr.client.kv;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.ByteString;

import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.concurrent.CompletableFutures;

/**
 * @author Stepan Koltsov
 *
 * @url https://wiki.yandex-team.ru/solomon/kikimr/KeyValueStorage
 *
 * @see KikimrKvClientSync
 */
@ParametersAreNonnullByDefault
public interface KikimrKvClient extends AutoCloseable {

    // so message could be read in messagebus response
    // see also https://st.yandex-team.ru/KIKIMR-904
    int DO_NOT_EXCEED_FILE_SIZE = 20 << 20;

    void setDoNotExceedFileSizeForTest(int doNotExceedFileSize);

    int getDoNotExceedFileSize();

    @Override
    void close();

    CompletableFuture<Void> createKvTablets(String path, int count);

    CompletableFuture<Void> alterKvTablets(String path, int count);

    CompletableFuture<Void> dropKvTablets(String path);

    /**
     * @return array of KV-tablet ids associated with the given scheme element
     *         or an empty array if according path does not exists.
     */
    CompletableFuture<long[]> resolveKvTablets(String path);

    @CheckReturnValue
    CompletableFuture<long[]> findTabletsOnLocalhost();

    @CheckReturnValue
    CompletableFuture<Long> incrementGeneration(long tabletId, long expiredAt);

    class Rename {
        private final String from;
        private final String to;

        public Rename(String from, String to) {
            this.from = from;
            this.to = to;
        }

        public String getFrom() {
            return from;
        }

        public String getTo() {
            return to;
        }
    }

    class Write {
        private final String name;
        @Nonnull
        private final ByteString value;
        private final MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel;
        private final MsgbusKv.TKeyValueRequest.EPriority priority;

        public static final MsgbusKv.TKeyValueRequest.EPriority defaultPriority =
            MsgbusKv.TKeyValueRequest.EPriority.REALTIME;

        public Write(String name, byte[] value,
            MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel, MsgbusKv.TKeyValueRequest.EPriority priority)
        {
            this.name = name;
            this.storageChannel = storageChannel;
            this.value = ByteString.copyFrom(value);
            this.priority = priority;
        }

        public Write(String name, ByteString value,
            MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel, MsgbusKv.TKeyValueRequest.EPriority priority)
        {
            this.name = name;
            this.value = value;
            this.storageChannel = storageChannel;
            this.priority = priority;
        }

        public String getName() {
            return name;
        }

        @Nonnull
        public ByteString getValue() {
            return value;
        }

        @Nonnull
        public MsgbusKv.TKeyValueRequest.EStorageChannel getStorageChannel() {
            return storageChannel;
        }

        @Nonnull
        public MsgbusKv.TKeyValueRequest.EPriority getPriority() {
            return priority;
        }

        @Override
        public String toString() {
            return "Write{" +
                "name='" + name + '\'' +
                ", value=" + value +
                ", storageChannel=" + storageChannel +
                ", priority=" + priority +
                '}';
        }
    }

    class Concat {
        private final List<String> inputs;
        private final String output;
        private final boolean keepInputs;

        public Concat(List<String> inputs, String output, boolean keepInputs) {
            this.inputs = inputs;
            this.output = output;
            this.keepInputs = keepInputs;
        }

        public List<String> getInputs() {
            return inputs;
        }

        public String getOutput() {
            return output;
        }

        public boolean isKeepInputs() {
            return keepInputs;
        }
    }

    @CheckReturnValue
    CompletableFuture<Void> writeAndRenameAndDeleteAndConcat(
        long tabletId,
        long gen,
        List<Write> writes,
        List<Rename> renames,
        List<NameRange> deletes,
        List<Concat> concats,
        long expiredAt);

    @CheckReturnValue
    default CompletableFuture<Void> writeAndRenameAndDelete(
        long tabletId,
        long gen,
        List<Write> writes,
        List<Rename> renames,
        List<NameRange> deletes,
        long expiredAt)
    {
        return writeAndRenameAndDeleteAndConcat(
            tabletId, gen,
            writes, renames, deletes, List.of(), expiredAt);
    }

    @CheckReturnValue
    default CompletableFuture<Void> writeAndRename(
        long tabletId,
        long gen,
        List<Write> writes,
        List<Rename> renames,
        long expiredAt)
    {
        return writeAndRenameAndDelete(tabletId, gen, writes, renames, List.of(), expiredAt);
    }

    @CheckReturnValue
    default CompletableFuture<Void> write(
        long tabletId, long gen, String key, byte[] value,
        MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel, MsgbusKv.TKeyValueRequest.EPriority priority, long expiredAt)
    {
        return writeAndRename(tabletId, gen, List.of(new Write(key, value, storageChannel, priority)), List.of(), expiredAt);
    }

    @CheckReturnValue
    default CompletableFuture<Void> write(
        long tabletId, long gen, String key, ByteString value,
        MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel, MsgbusKv.TKeyValueRequest.EPriority priority, long expiredAt)
    {
        return writeAndRename(tabletId, gen, List.of(new Write(key, value, storageChannel, priority)), List.of(), expiredAt);
    }

    @CheckReturnValue
    default CompletableFuture<Void> writeMulti(long tabletId, long gen, List<Write> writes, long expiredAt) {
        return writeAndRenameAndDelete(tabletId, gen, writes, List.of(), List.of(), expiredAt);
    }

    @CheckReturnValue
    default CompletableFuture<Void> deleteRange(long tabletId, long gen, NameRange nameRange, long expiredAt) {
        return deleteRanges(tabletId, gen, List.of(nameRange), expiredAt);
    }

    @CheckReturnValue
    default CompletableFuture<Void> deleteRanges(long tabletId, long gen, List<NameRange> nameRanges, long expiredAt) {
        return writeAndRenameAndDelete(tabletId, gen, List.of(), List.of(), nameRanges, expiredAt);
    }

    @CheckReturnValue
    default CompletableFuture<Void> deleteFiles(long tabletId, long gen, List<String> names, long expiredAt) {
        List<NameRange> ranges = names.stream().map(NameRange::single).collect(Collectors.toList());
        return deleteRanges(tabletId, gen, ranges, expiredAt);
    }

    @CheckReturnValue
    CompletableFuture<Void> copyRange(long tabletId, long gen, NameRange nameRange, String addPrefix, String removePrefix, long expiredAt);

    default CompletableFuture<Void> concatAndDeleteOriginals(long tabletId, long gen, List<String> inputs, String output, long expiredAt) {
        return writeAndRenameAndDeleteAndConcat(tabletId, gen,
            List.of(), List.of(), List.of(), List.of(new Concat(inputs, output, false)), expiredAt);
    }

    class KvEntry {
        @Nonnull
        private final String name;
        @Nonnull
        private final byte[] value;

        public KvEntry(@Nonnull String name, @Nonnull byte[] value) {
            this.name = name;
            this.value = value;
        }

        @Nonnull
        public String getName() {
            return name;
        }

        @Nonnull
        public byte[] getValue() {
            return value;
        }

        public void checkValueIsEmpty() {
            if (value.length > 0) {
                throw new IllegalStateException();
            }
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            KvEntry kvEntry = (KvEntry) o;

            if (!name.equals(kvEntry.name)) return false;
            return Arrays.equals(value, kvEntry.value);

        }

        @Override
        public int hashCode() {
            int result = name.hashCode();
            result = 31 * result + Arrays.hashCode(value);
            return result;
        }

        @Override
        public String toString() {
            return name + ":" + Hex.encodeHr(value);
        }
    }

    class KvEntryWithStats {
        @Nonnull
        private final String name;
        @Nonnull
        private final byte[] value;
        private final int size;
        private final long createdUnixtime;

        public KvEntryWithStats(@Nonnull String name, @Nonnull byte[] value, int size, long createdUnixtime) {
            if (value.length > 0) {
                if (value.length != size) {
                    throw new IllegalArgumentException();
                }
            }

            this.name = name;
            this.value = value;
            this.size = size;
            this.createdUnixtime = createdUnixtime;
        }

        @Nonnull
        public KvEntryWithStats withUnixtime(int createdUnixtime) {
            return new KvEntryWithStats(name, value, size, createdUnixtime);
        }

        public void checkValueIsEmpty() {
            if (value.length > 0) {
                throw new IllegalStateException();
            }
        }

        @Nonnull
        public String getName() {
            return name;
        }

        @Nonnull
        public byte[] getValue() {
            return value;
        }

        public int getSize() {
            return size;
        }

        public KvEntryStats toStats() {
            return new KvEntryStats(name, size, createdUnixtime);
        }


        @Override
        public String toString() {
            return name
                + ":" + Hex.encodeHr(value)
                + ":" + size
                + ":" + KikimrKvClientInternal.toSecondsFormatter.format(Instant.ofEpochSecond(createdUnixtime));
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            KvEntryWithStats that = (KvEntryWithStats) o;

            if (size != that.size) return false;
            if (createdUnixtime != that.createdUnixtime) return false;
            if (!name.equals(that.name)) return false;
            return Arrays.equals(value, that.value);

        }

        @Override
        public int hashCode() {
            int result = name.hashCode();
            result = 31 * result + Arrays.hashCode(value);
            result = 31 * result + size;
            result = 31 * result + (int) (createdUnixtime ^ (createdUnixtime >>> 32));
            return result;
        }
    }

    class KvEntryStats {
        @Nonnull
        private final String name;
        private final int size;
        private final long createdUnixtime;

        public KvEntryStats(@Nonnull String name, int size, long createdUnixtime) {
            this.name = name;
            this.size = size;
            this.createdUnixtime = createdUnixtime;
        }

        @Nonnull
        public String getName() {
            return name;
        }

        public int getSize() {
            return size;
        }

        public long getCreatedUnixtime() {
            return createdUnixtime;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            KvEntryStats that = (KvEntryStats) o;
            if (size != that.size) return false;
            if (createdUnixtime != that.createdUnixtime) return false;
            return name.equals(that.name);
        }

        @Override
        public int hashCode() {
            int result = name.hashCode();
            result = 31 * result + size;
            result = 31 * result + (int) (createdUnixtime ^ (createdUnixtime >>> 32));
            return result;
        }

        @Override
        public String toString() {
            return name + ":" + size + ":" + KikimrKvClientInternal.toSecondsFormatter.format(Instant.ofEpochSecond(createdUnixtime));
        }
    }

    CompletableFuture<KvReadRangeResult> readRange(long tabletId, long gen, NameRange nameRange, boolean includeData, long limitBytes, long expiredAt);

    default CompletableFuture<KvReadRangeResult> readRange(long tabletId, long gen, NameRange nameRange, boolean includeData, long expiredAt) {
        return readRange(tabletId, gen, nameRange, includeData, KvRange.LEN_UNLIMITED, expiredAt);
    }

    default CompletableFuture<KvReadRangeResult> readRangeData(long tabletId, long gen, NameRange nameRange, long expiredAt) {
        return readRange(tabletId, gen, nameRange, true, expiredAt);
    }

    default CompletableFuture<List<KvEntryStats>> readRangeNames(long tabletId, long gen, NameRange nameRange, long expiredAt) {
        return readRange(tabletId, gen, nameRange, false, expiredAt)
            .thenApply(r -> {
                return Arrays.stream(r.getEntriesAll()).map(kvEntry -> {
                    kvEntry.checkValueIsEmpty();
                    return kvEntry.toStats();
                }).collect(Collectors.toList());
            });
    }

    default CompletableFuture<List<KvEntryStats>> readRangeNames(long tabletId, long gen, long expiredAt) {
        return readRangeNames(tabletId, gen, NameRange.all(), expiredAt);
    }

    default KvAsyncIterator<ArrayList<KvEntryWithStats>> readRangeAllIter(
        long tabletId, long gen, NameRange nameRange, long expiredAt)
    {
        return readRangeAllIter(tabletId, gen, nameRange, KvRange.LEN_UNLIMITED, expiredAt);
    }

    default KvAsyncIterator<ArrayList<KvEntryWithStats>> readRangeAllIter(
        long tabletId, long gen, NameRange nameRange, long limitBytes, long expiredAt)
    {
        return new KvAsyncIteratorReadRange(this, tabletId, gen, nameRange, limitBytes);
    }

    CompletableFuture<Optional<byte[]>> readData(long tabletId, long gen, String name, long offset, long length, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority);

    default CompletableFuture<Optional<byte[]>> readData(long tabletId, long gen, String name, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        return readData(tabletId, gen, name, 0, KvRange.LEN_UNLIMITED, expiredAt, priority);
    }

    default CompletableFuture<byte[]> readDataSome(long tabletId, long gen, String name, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        return readDataSome(tabletId, gen, name, 0, KvRange.LEN_UNLIMITED, expiredAt, priority);
    }

    default CompletableFuture<byte[]> readDataSome(long tabletId, long gen, String name, long offset, long length, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        return readData(tabletId, gen, name, offset, length, expiredAt, priority)
            .thenApply(o -> {
                return o.orElseThrow(() -> new RuntimeException("file not found: " + name));
            });
    }

    default CompletableFuture<Void> readDataLargeImpl(
        long tabletId, long gen, String name, long offset, long length, ByteArrayOutputStream os, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority)
    {
        if (length == 0) {
            throw new IllegalArgumentException();
        }
        KvRange.validateLen(length);

        int doNotExceedFileSize = getDoNotExceedFileSize();

        long toRead;
        if (length == KvRange.LEN_UNLIMITED) {
            toRead = doNotExceedFileSize;
        } else {
            toRead = Math.min(doNotExceedFileSize, length);
        }

        return readDataSome(tabletId, gen, name, offset, toRead, expiredAt, priority)
            .thenCompose(r -> {
                if (r.length == 0) {
                    if (length != KvRange.LEN_UNLIMITED) {
                        throw new RuntimeException("empty result with non-empty remaining size");
                    }
                    return CompletableFuture.completedFuture(null);
                } else {
                    try {
                        os.write(r);
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                    if (length == KvRange.LEN_UNLIMITED) {
                        if (r.length == doNotExceedFileSize) {
                            return readDataLargeImpl(
                                tabletId,
                                gen,
                                name,
                                offset + doNotExceedFileSize, KvRange.LEN_UNLIMITED,
                                os, expiredAt, priority);
                        } else {
                            return CompletableFuture.completedFuture(null);
                        }
                    } else {
                        if (length != r.length) {
                            if (r.length != toRead) {
                                throw new RuntimeException("truncated");
                            }
                            return readDataLargeImpl(
                                tabletId,
                                gen,
                                name,
                                offset + r.length,
                                length - r.length,
                                os, expiredAt, priority);
                        } else {
                            return CompletableFuture.completedFuture(null);
                        }
                    }
                }
            });
    }

    @CheckReturnValue
    default CompletableFuture<byte[]> readDataLarge(long tabletId, long gen, String name, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        return readDataLarge(tabletId, gen, 0, KvRange.LEN_UNLIMITED, name, expiredAt, priority);
    }

    @CheckReturnValue
    default CompletableFuture<byte[]> readDataLarge(long tabletId, long gen, long offset, long len, String name, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        return readDataLargeImpl(tabletId, gen, name, offset, len, os, expiredAt, priority).thenApply(u -> os.toByteArray());
    }

    @CheckReturnValue
    default CompletableFuture<byte[][]> readDataLargeMulti(long tabletId, long gen, KvChunkAddress[] addresses, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        // TODO: read sequentially
        List<CompletableFuture<byte[]>> futures = Arrays.stream(addresses)
            .map(a -> readDataLarge(tabletId, gen, a.getOffset(), a.getLength(), a.getName(), expiredAt, priority))
            .collect(Collectors.toList());

        return CompletableFutures.allOf(futures)
            .thenApply(l -> l.toArray(new byte[0][]));
    }

    /**
     * @return empty arrays for missing files
     */
    default CompletableFuture<byte[][]> readDataMulti(long tabletId, long gen, KvChunkAddress[] addresses, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        int knownSumLength = Arrays.stream(addresses)
            .mapToInt(KvChunkAddress::getLength)
            .filter(l -> l >= 0)
            .sum();

        if (knownSumLength > getDoNotExceedFileSize()) {
            throw new IllegalArgumentException(
                "cannot read so large result: " + knownSumLength
                    + " in single request: " + Arrays.stream(addresses));
        }

        List<CompletableFuture<Optional<byte[]>>> futures = Arrays.stream(addresses)
            .map(a -> readData(tabletId, gen, a.getName(), a.getOffset(), a.getLength(), expiredAt, priority))
            .collect(Collectors.toList());
        return CompletableFutures.allOf(futures)
            .thenApply(obs -> obs.stream()
                .map(bytes -> bytes.orElseGet(() -> new byte[0]))
                .toArray(byte[][]::new));
    }

    @CheckReturnValue
    default CompletableFuture<Void> rename(long tabletId, long gen, String from, String to, long expiredAt) {
        return renameMultiple(tabletId, gen, List.of(new Rename(from, to)), expiredAt);
    }

    @CheckReturnValue
    default CompletableFuture<Void> renameMultiple(long tabletId, long gen, List<Rename> renames, long expiredAt) {
        return writeAndRenameAndDelete(tabletId, gen, List.of(), renames, List.of(), expiredAt);
    }

}
