package ru.yandex.kikimr.client.kv;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.kikimr.client.KikimrAnyResponseException;
import ru.yandex.kikimr.client.KikimrException;
import ru.yandex.kikimr.client.KikimrTransport;
import ru.yandex.kikimr.client.ReplyStatus;
import ru.yandex.kikimr.client.ResponseStatus;
import ru.yandex.kikimr.proto.FlatSchemeOp;
import ru.yandex.kikimr.proto.FlatTxScheme;
import ru.yandex.kikimr.proto.Msgbus;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.proto.Tablet;
import ru.yandex.kikimr.util.NameRange;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class KikimrKvClientImpl implements KikimrKvClient, AutoCloseable {

    private static final Logger logger = LoggerFactory.getLogger(KikimrKvClientImpl.class);

    private final KikimrTransport transport;

    public KvCounters counters = new KvCounters();
    private int doNotExceedFileSize = DO_NOT_EXCEED_FILE_SIZE;

    public KikimrKvClientImpl(KikimrTransport transport) {
        this.transport = transport;
    }

    @Override
    public void setDoNotExceedFileSizeForTest(int doNotExceedFileSize) {
        this.doNotExceedFileSize = doNotExceedFileSize;
    }

    @Override
    public int getDoNotExceedFileSize() {
        return doNotExceedFileSize;
    }

    @Override
    @PreDestroy
    public void close() {
        transport.close();
    }

    private static final int DOMAIN_UID = 1;
    private static final int CHANNEL_PROFILE_ID = 0;
    private static final Tablet.TTabletTypes.EType KV_TABLET_FLAT_TYPE = Tablet.TTabletTypes.EType.KeyValue;

    @Override
    public CompletableFuture<Void> createKvTablets(String pathStr, int count) {
        Msgbus.TSchemeOperation.Builder req = Msgbus.TSchemeOperation.newBuilder();
        FlatSchemeOp.TModifyScheme.Builder modifyScheme = req.getTransactionBuilder().getModifySchemeBuilder();
        Path path = Path.of(pathStr);
        modifyScheme.setWorkingDir(path.getParent().toString());
        modifyScheme.setOperationType(FlatSchemeOp.EOperationType.ESchemeOpCreateSolomonVolume);

        FlatSchemeOp.TCreateSolomonVolume.Builder solomonVolume = modifyScheme.getCreateSolomonVolumeBuilder();
        solomonVolume.setName(path.getFileName().toString());
        solomonVolume.setPartitionCount(count);
        solomonVolume.setChannelProfileId(CHANNEL_PROFILE_ID);

        return transport.schemeOperation(req.build())
            .thenCompose(this::waitForResponse)
            .thenAccept(resp -> {
                if (resp.getStatus() != ResponseStatus.MSTATUS_OK.cppValue()) {
                    throw new KikimrAnyResponseException(resp, "create table");
                }
            });
    }

    @Override
    public CompletableFuture<Void> alterKvTablets(String pathStr, int count) {
        Msgbus.TSchemeOperation.Builder req = Msgbus.TSchemeOperation.newBuilder();
        FlatSchemeOp.TModifyScheme.Builder modifyScheme = req.getTransactionBuilder().getModifySchemeBuilder();
        Path path = Path.of(pathStr);
        modifyScheme.setWorkingDir(path.getParent().toString());
        modifyScheme.setOperationType(FlatSchemeOp.EOperationType.ESchemeOpAlterSolomonVolume);

        FlatSchemeOp.TAlterSolomonVolume.Builder solomonVolume = modifyScheme.getAlterSolomonVolumeBuilder();
        solomonVolume.setName(path.getFileName().toString());
        solomonVolume.setPartitionCount(count);
        solomonVolume.setChannelProfileId(CHANNEL_PROFILE_ID);

        return transport.schemeOperation(req.build())
                .thenCompose(this::waitForResponse)
                .thenAccept(resp -> {
                    if (resp.getStatus() != ResponseStatus.MSTATUS_OK.cppValue()) {
                        throw new KikimrAnyResponseException(resp, "create table");
                    }
                });
    }

    @Override
    public CompletableFuture<Void> dropKvTablets(String pathStr) {
        Msgbus.TSchemeOperation.Builder req = Msgbus.TSchemeOperation.newBuilder();
        FlatSchemeOp.TModifyScheme.Builder modifyScheme = req.getTransactionBuilder().getModifySchemeBuilder();
        Path path = Path.of(pathStr);
        modifyScheme.setWorkingDir(path.getParent().toString());
        modifyScheme.setOperationType(FlatSchemeOp.EOperationType.ESchemeOpDropSolomonVolume);
        modifyScheme.getDropBuilder().setName(path.getFileName().toString());

        return transport.schemeOperation(req.build())
            .thenCompose(this::waitForResponse)
            .thenAccept(resp -> {
                if (resp.getStatus() != ResponseStatus.MSTATUS_OK.cppValue()) {
                    throw new KikimrAnyResponseException(resp, "drop kv");
                }
            });
    }

    private CompletableFuture<Msgbus.TResponse> waitForResponse(Msgbus.TResponse response) {
        if (response.getStatus() != ResponseStatus.MSTATUS_INPROGRESS.cppValue()) {
            return CompletableFuture.completedFuture(response);
        }

        Msgbus.TFlatTxPollOptions pollOptions = Msgbus.TFlatTxPollOptions.newBuilder()
            .setTimeout(10_000) // 10 seconds
            .build();
        Msgbus.TSchemeOperationStatus request = Msgbus.TSchemeOperationStatus.newBuilder()
            .setFlatTxId(response.getFlatTxId())
            .setPollOptions(pollOptions)
            .build();

        return transport.schemeOperationStatus(request)
            .thenCompose(this::waitForResponse);
    }

    @Override
    public CompletableFuture<long[]> resolveKvTablets(String path) {
        Msgbus.TSchemeDescribe.Builder r = Msgbus.TSchemeDescribe.newBuilder();
        r.setPath(path);
        return transport.schemeDescribe(r.build())
            .thenApply(resp -> {
                if (resp.getStatus() == ResponseStatus.MSTATUS_OK.cppValue()) {
                    FlatSchemeOp.TPathDescription pathDescription = resp.getPathDescription();
                    FlatSchemeOp.TSolomonVolumeDescription solomonDescription = pathDescription.getSolomonDescription();
                    return solomonDescription.getPartitionsList().stream()
                        .mapToLong(FlatSchemeOp.TSolomonVolumeDescription.TPartition::getTabletId)
                        .toArray();
                }
                if (resp.getSchemeStatus() == FlatTxScheme.EStatus.StatusPathDoesNotExist.getNumber()) {
                    return new long[0];
                }
                throw new KikimrAnyResponseException(resp, "describe");
            });
    }

    @Override
    public CompletableFuture<long[]> findTabletsOnLocalhost() {
        Msgbus.TLocalEnumerateTablets.Builder request = Msgbus.TLocalEnumerateTablets.newBuilder();

        request.setDomainUid(DOMAIN_UID);
        request.setTabletType(KV_TABLET_FLAT_TYPE);

        return transport.localEnumerateTablets(request.build())
            .thenApply(m -> {
                if (m.getStatus() != ResponseStatus.MSTATUS_OK.cppValue()) {
                    throw new KikimrAnyResponseException(m, "LocalEnumerateTablets");
                }

                return m.getTabletInfoList().stream().mapToLong(t -> {
                    long tabletId = t.getTabletId();
                    if (tabletId == 0) {
                        throw new IllegalStateException();
                    }
                    return tabletId;
                }).toArray();
            });
    }

    @Override
    public CompletableFuture<Long> incrementGeneration(long tabletId, long expiredAt) {
        MsgbusKv.TKeyValueRequest.Builder request = MsgbusKv.TKeyValueRequest.newBuilder();

        request.setTabletId(tabletId);
        request.getCmdIncrementGenerationBuilder();
        if (expiredAt > 0) {
            request.setDeadlineInstantMs(expiredAt);
        }

        return transport.keyValue(request.build()).thenApply(r -> {
            if (!ResponseStatus.isSuccess(r.getStatus(), false)) {
                throw new KikimrAnyResponseException(r, "increment generation", r.getErrorReason());
            }

            if (!r.hasIncrementGenerationResult()) {
                throw new KikimrException();
            }

            MsgbusKv.TKeyValueResponse.TIncrementGenerationResult igr = r.getIncrementGenerationResult();

            ReplyStatus.checkOk(igr.getStatus(), request, igr);

            if (igr.getGeneration() != 0) {
                return igr.getGeneration();
            }

            throw new KikimrException("unknown response");
        });
    }

    private CompletableFuture<Void> kvSendCommands(
        long tabletId, long gen, List<? extends KvCommandUntyped> commands, long expiredAt)
    {
        MsgbusKv.TKeyValueRequest.Builder request = MsgbusKv.TKeyValueRequest.newBuilder();
        if (expiredAt > 0) {
            request.setDeadlineInstantMs(expiredAt);
        }
        request.setTabletId(tabletId);
        if (gen != 0) {
            request.setGeneration(gen);
        }

        for (KvCommandUntyped command : commands) {
            command.addTo(request);
        }

        return transport.keyValue(request.build())
            .thenAccept(r -> {
                if (r.getStatus() == ResponseStatus.MSTATUS_REJECTED.cppValue()) {
                    logger.error("generation of tablet " + tabletId + " changed: " + r.getErrorReason());
                    throw new KikimrKvGenerationChangedRuntimeException(tabletId);
                }

                if (!ResponseStatus.isSuccess(r.getStatus(), false)) {
                    String op = MsgbusKv.TKeyValueRequest.getDescriptor().getName();
                    throw new KikimrAnyResponseException(r, op, r.getErrorReason());
                }

                for (KvCommandUntyped command : commands) {
                    command.checkResponse(r, counters);
                }
            });
    }

    private CompletableFuture<Void> kvSendCommand(long tabletId, long gen, KvCommandUntyped command, long expiredAt) {
        return kvSendCommands(tabletId, gen, List.of(command), expiredAt);
    }

    @Override
    public CompletableFuture<KvReadRangeResult> readRange(long tabletId, long gen, NameRange nameRange, boolean includeData, long limitBytes, long expiredAt) {
        KvCommandReadRange command = new KvCommandReadRange(nameRange, includeData, limitBytes);

        return kvSendCommand(tabletId, gen, command, expiredAt)
            .thenApply(u -> command.getResult());
    }

    @Override
    public CompletableFuture<Optional<byte[]>> readData(long tabletId, long gen, String name, long offset, long length, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        KvCommandReadOne command = new KvCommandReadOne(name, offset, length, priority);
        return kvSendCommand(tabletId, gen, command, expiredAt)
            .thenApply(u -> command.getData());
    }

    @Override
    public CompletableFuture<byte[][]> readDataMulti(long tabletId, long gen, KvChunkAddress[] addresses, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        List<KvCommandReadOne> commands = Arrays.stream(addresses)
            .map(a -> new KvCommandReadOne(a.getName(), a.getOffset(), a.getLength(), priority))
            .collect(Collectors.toList());

        return kvSendCommands(tabletId, gen, commands, expiredAt)
            .thenApply(u -> {
                return commands.stream()
                    .map(r -> r.getData().orElseGet(() -> new byte[0]))
                    .toArray(byte[][]::new);
            });
    }

    @Override
    public CompletableFuture<Void> writeAndRenameAndDeleteAndConcat(
        long tabletId, long gen,
        List<Write> writes,
        List<Rename> renames,
        List<NameRange> deletes,
        List<Concat> concats, long expiredAt)
    {
        List<KvCommandUntyped> commands = new ArrayList<>(writes.size() + renames.size() + deletes.size());

        for (Write write : writes) {
            if (write.getValue().size() > doNotExceedFileSize) {
                throw new IllegalArgumentException("too large file: " + write + "; limit: " + doNotExceedFileSize);
            }

            commands.add(new KvCommandWrite(
                write.getName(), write.getValue(), write.getStorageChannel(), write.getPriority()));
        }
        for (Rename rename : renames) {
            commands.add(new KvCommandRename(rename.getFrom(), rename.getTo()));
        }
        for (NameRange delete : deletes) {
            commands.add(new KvCommandDeleteRange(delete));
        }
        for (Concat concat : concats) {
            commands.add(new KvCommandConcat(concat.getInputs(), concat.getOutput(), concat.isKeepInputs()));
        }

        return kvSendCommands(tabletId, gen, commands, expiredAt);
    }

    @Override
    public CompletableFuture<Void> copyRange(long tabletId, long gen, NameRange nameRange, String addPrefix, String removePrefix, long expiredAt) {
        return kvSendCommand(tabletId, gen, new KvCommandCloneRange(nameRange, addPrefix, removePrefix), expiredAt);
    }
}
