package ru.yandex.kikimr.client.kv;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillClose;

import com.google.protobuf.ByteString;

import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class KikimrKvClientSync implements AutoCloseable {

    final KikimrKvClient kikimrKvClient;

    public KikimrKvClientSync(@WillClose KikimrKvClient kikimrKvClient) {
        this.kikimrKvClient = kikimrKvClient;
    }

    public List<KikimrKvClient.KvEntryStats> kvReadRangeNames(long tabletId, long gen, NameRange nameRange) {
        return kikimrKvClient.readRangeNames(tabletId, gen, nameRange, 0).join();
    }

    public long[] findTabletsOnLocalhost() {
        return kikimrKvClient.findTabletsOnLocalhost().join();
    }

    public void createKvTablesWithSchemeShard(String path, int count) {
        kikimrKvClient.createKvTablets(path, count).join();
    }

    public long[] resolveKvTablesWithSchemeShard(String path) {
        return kikimrKvClient.resolveKvTablets(path).join();
    }

    public long incrementGeneration(long tabletId) {
        return kikimrKvClient.incrementGeneration(tabletId, 0).join();
    }

    public void deleteAll(long tabletId, long gen) {
        kikimrKvClient.deleteRange(tabletId, gen, NameRange.all(), 0).join();
    }

    public void deleteRange(long tabletId, long gen, NameRange nameRange) {
        kikimrKvClient.deleteRange(tabletId, gen, nameRange, 0).join();
    }

    public Optional<byte[]> readData(long tabletId, long gen, String name, long offset, long length) {
        return kikimrKvClient.readData(tabletId, gen, name, offset, length, 0, MsgbusKv.TKeyValueRequest.EPriority.REALTIME).join();
    }

    public Optional<byte[]> readData(long tabletId, long gen, String name) {
        return readData(tabletId, gen, name, 0, -1);
    }

    public byte[] readDataLarge(long tabletId, long gen, String name) {
        return kikimrKvClient.readDataLarge(tabletId, gen, name, 0, MsgbusKv.TKeyValueRequest.EPriority.REALTIME).join();
    }

    @Nonnull
    public byte[] readDataSome(long tabletId, long gen, String name) {
        return readData(tabletId, gen, name).orElseThrow(() -> new RuntimeException("file not found: " + name));
    }

    @Nonnull
    public byte[] readDataSome(long tabletId, long gen, String name, long offset, long length) {
        return readData(tabletId, gen, name, offset, length).orElseThrow(() -> new RuntimeException("file not found: " + name));
    }

    public boolean readDataExists(long tabletId, long gen, String name) {
        return readData(tabletId, gen, name).isPresent();
    }

    public byte[][] readDataMulti(long tabletId, long gen, KvChunkAddress[] addresses) {
        return kikimrKvClient.readDataMulti(tabletId, gen, addresses, 0, MsgbusKv.TKeyValueRequest.EPriority.REALTIME).join();
    }

    public List<KikimrKvClient.KvEntryStats> readRangeNames(long tabletId, long gen) {
        return kvReadRangeNames(tabletId, gen, NameRange.all());
    }

    public KvReadRangeResult readRange(long tabletId, long gen, NameRange nameRange, boolean includeData) {
        return kikimrKvClient.readRange(tabletId, gen, nameRange, includeData, 0).join();
    }

    public KvReadRangeResult readRange(long tabletId, long gen, NameRange nameRange, boolean includeData, long limitBytes) {
        return kikimrKvClient.readRange(tabletId, gen, nameRange, includeData, limitBytes, 0).join();
    }

    public void rename(long tabletId, long gen, String from, String to) {
        kikimrKvClient.rename(tabletId, gen, from, to, 0).join();
    }

    public void writeMulti(long tabletId, long gen, List<KikimrKvClient.Write> writes) {
        kikimrKvClient.writeMulti(tabletId, gen, writes, 0).join();
    }

    public void write(
        long tabletId, long gen, String key, byte[] value,
        MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel, MsgbusKv.TKeyValueRequest.EPriority priority)
    {
        kikimrKvClient.write(tabletId, gen, key, value, storageChannel, priority, 0).join();
    }

    public void writeDefault(
        long tabletId, long gen, String key, byte[] value)
    {
        write(tabletId, gen, key, value,
            MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN, KikimrKvClient.Write.defaultPriority);
    }

    public void writeLarge(
        long tabletId, long gen, String key, byte[] value,
        Supplier<String> tempChunkFileNames,
        MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel, MsgbusKv.TKeyValueRequest.EPriority priority)
    {
        if (value.length < KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE) {
            write(tabletId, gen, key, value, storageChannel, priority);
        } else {
            List<String> names = new ArrayList<>();

            for (int i = 0; i < (value.length + KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE - 1) / KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE; ++i) {
                String partName = tempChunkFileNames.get();
                int offset = i * KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE;
                int length = Math.min(KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE, value.length - offset);
                ByteString partBytes = ByteString.copyFrom(value, offset, length);

                write(tabletId, gen, partName, partBytes.toByteArray(), storageChannel, priority);
                names.add(partName);
            }
            kikimrKvClient.concatAndDeleteOriginals(tabletId, 0, names, key, 0).join();
        }
    }

    @Override
    public void close() {
        kikimrKvClient.close();
    }
}
