package ru.yandex.kikimr.client.kv;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.LongStream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.ByteString;
import com.yandex.ydb.yndx.rate_limiter.AcquireLockResult;

import ru.yandex.kikimr.client.kv.commands.Command;
import ru.yandex.kikimr.client.kv.commands.CopyRange;
import ru.yandex.kikimr.client.kv.commands.KeyRange;
import ru.yandex.kikimr.client.kv.commands.Priority;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;

/**
 * @author senyasdr
 */
@ParametersAreNonnullByDefault
public class KikimrKvV2ClientImpl implements KikimrKvClient {

    private volatile int doNotExceedFileSize = DO_NOT_EXCEED_FILE_SIZE;
    private final KikimrV2NodeRouter nodeRouter;
    private final String path;

    public KikimrKvV2ClientImpl(KikimrV2NodeRouter nodeRouter, String path) {
        this.nodeRouter = nodeRouter;
        this.path = path;
    }

    @Override
    public void setDoNotExceedFileSizeForTest(int doNotExceedFileSize) {
        this.doNotExceedFileSize = doNotExceedFileSize;
    }

    @Override
    public int getDoNotExceedFileSize() {
        return doNotExceedFileSize;
    }

    @Override
    public void close() {
        nodeRouter.close();
    }

    @Override
    public CompletableFuture<Void> createKvTablets(String path, int count) {
        return nodeRouter.createKvVolume(path, count).thenAccept(r -> {
        });
    }

    @Override
    public CompletableFuture<Void> alterKvTablets(String path, int count) {
        return nodeRouter.alterVolume(path, count).thenAccept(r -> {
        });
    }

    @Override
    public CompletableFuture<Void> dropKvTablets(String path) {
        return nodeRouter.dropKvVolume(path).thenAccept(r -> {
        });
    }

    @Override
    public CompletableFuture<long[]> resolveKvTablets(String path) {
        return nodeRouter.describeVolume(path)
                .thenApply(r -> LongStream.range(0, r.getPartitionCount()).toArray());
    }

    @Override
    public CompletableFuture<long[]> findTabletsOnLocalhost() {
        return nodeRouter.listLocalPartitions(0, path)
                .thenApply(r -> r
                        .getPartitionIdsList()
                        .stream()
                        .mapToLong(l -> l)
                        .toArray());
    }

    @Override
    public CompletableFuture<Long> incrementGeneration(long parttionId, long expiredAt) {
        return nodeRouter.acquireLock(parttionId, path, expiredAt)
                .thenApply(AcquireLockResult::getLockGeneration);
    }

    @Override
    public CompletableFuture<Void> writeAndRenameAndDeleteAndConcat(long partitionId, long gen, List<Write> writes, List<Rename> renames, List<NameRange> deletes, List<Concat> concats, long expiredAt) {
        List<Command> commands = new ArrayList<>();
        writes.forEach(c -> commands.add(ru.yandex.kikimr.client.kv.commands.Write.fromV1Command(c)));
        renames.forEach(c -> commands.add(ru.yandex.kikimr.client.kv.commands.Rename.fromV1Command(c)));
        deletes.forEach(c -> commands.add(ru.yandex.kikimr.client.kv.commands.DeleteRange.fromV1Command(c)));
        concats.forEach(c -> commands.add(ru.yandex.kikimr.client.kv.commands.Concat.fromV1Command(c)));
        return nodeRouter.executeTransaction(commands, path, partitionId, gen).thenAccept(r -> {
        });
    }

    @Override
    public CompletableFuture<Void> copyRange(long partitionId, long gen, NameRange nameRange, String addPrefix, String removePrefix, long expiredAt) {
        List<Command> copyRange = List.of(new CopyRange(new KeyRange(nameRange), removePrefix, addPrefix));
        return nodeRouter.executeTransaction(copyRange, path, partitionId, gen).thenAccept(r -> {
        });
    }

    @Override
    public CompletableFuture<KvReadRangeResult> readRange(long partitionId, long gen, NameRange nameRange, boolean includeData, long limitBytes, long expiredAt) {
        KeyRange range = new KeyRange(nameRange);
        if (includeData) {
            return nodeRouter.readRange(path, partitionId, range, gen, limitBytes, Priority.PRIORITY_UNSPECIFIED)
                    .thenApply(r -> KvReadRangeResult.fromReadRange(r.getPairList(), r.getIsOverrun()));
        }
        return nodeRouter.listRange(path, partitionId, range, gen, limitBytes)
                .thenApply(r -> KvReadRangeResult.fromListRange(r.getKeyList(), r.getIsOverrun()));
    }

    @Override
    public CompletableFuture<Optional<byte[]>> readData(long partitionId, long gen, String name, long offset, long length, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        return nodeRouter.read(path, partitionId, name, gen, offset, length, length)
                .thenApply(r -> Optional.of(r.getValue()).filter(b -> !b.isEmpty()).map(ByteString::toByteArray));
    }
}
