package ru.yandex.kikimr.client.kv;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.yndx.rate_limiter.AcquireLockResult;
import com.yandex.ydb.yndx.rate_limiter.DescribeVolumeResult;
import com.yandex.ydb.yndx.rate_limiter.ExecuteTransactionResult;
import com.yandex.ydb.yndx.rate_limiter.ListLocalPartitionsResult;
import com.yandex.ydb.yndx.rate_limiter.ListRangeResult;
import com.yandex.ydb.yndx.rate_limiter.ReadRangeResult;
import com.yandex.ydb.yndx.rate_limiter.ReadResult;

import ru.yandex.kikimr.client.KikimrException;
import ru.yandex.kikimr.client.kv.commands.Command;
import ru.yandex.kikimr.client.kv.commands.KeyRange;
import ru.yandex.kikimr.client.kv.commands.Priority;
import ru.yandex.kikimr.client.kv.noderesolver.KikmirV2NodeResolver;
import ru.yandex.kikimr.client.kv.noderesolver.Node;

/**
 * @author senyasdr
 */
public class KikimrV2NodeRouter implements AutoCloseable {

    private final KikmirV2NodeResolver nodeResolver;

    public KikimrV2NodeRouter(KikmirV2NodeResolver nodeResolver) {
        this.nodeResolver = nodeResolver;
    }

    public CompletableFuture<ListLocalPartitionsResult> listLocalPartitions(int nodeId, String path) {
        var listLocalPartitions = nodeResolver.getLocalNode().listLocalPartitions(nodeId, path);
        return listLocalPartitions.thenApply(r -> {
            ListLocalPartitionsResult result = r.expect("listLocalPartition");
            if (result.getNodeId() != 0) {
                throw new KikimrException("The specified path does not exist on the local node. The response returned from the node:" + result.getNodeId());
            }
            return result;
        });
    }

    public CompletableFuture<Void> createKvVolume(String path, int count) {
        return nodeResolver.getRandomNode().createKvVolume(path, count).thenAccept(r -> r.expect("dropKvVolume"));
    }

    public CompletableFuture<Void> dropKvVolume(String path) {
        return nodeResolver.getRandomNode().dropKvVolume(path).thenAccept(r -> r.expect("dropKvVolume"));
    }

    public CompletableFuture<Void> alterVolume(String path, int newPartitionCount) {
        return nodeResolver.getRandomNode().alterVolume(path, newPartitionCount)
                .thenAccept(r -> r.expect("alterVolume"));
    }

    public CompletableFuture<DescribeVolumeResult> describeVolume(String path) {
        return nodeResolver.getRandomNode().describeVolume(path).thenApply(r -> r.expect("describeVolume"));
    }

    public CompletableFuture<AcquireLockResult> acquireLock(long partitionId, String path, long expiredAt) {
        Node node = nodeResolver.getNodeIdByPathAndPartitionIdOrRandom(path, partitionId);
        return node.acquireLock(partitionId, path, expiredAt).thenApply(r -> {
            AcquireLockResult result = r.expect("acquireLock");
            if (result.getNodeId() != 0) {
                nodeResolver.updateNodeId(path, partitionId, result.getNodeId());
            }
            return result;
        });
    }

    public CompletableFuture<ReadResult> read(String path, long partitionId, String key, long gen, long offset, long size, long limitBytes) {
        return nodeResolver.getNodeIdByPathAndPartitionIdOrRandom(path, partitionId)
                .read(path, partitionId, key, gen, offset, size, limitBytes)
                .thenApply(r -> {
                    if (StatusCode.PRECONDITION_FAILED.equals(r.getCode())) {
                        throw new KikimrKvGenerationChangedRuntimeException(partitionId);
                    }
                    ReadResult result = r.expect("read");
                    if (result.getNodeId() != 0) {
                        nodeResolver.updateNodeId(path, partitionId, result.getNodeId());
                    }
                    return result;
                });
    }

    public CompletableFuture<ExecuteTransactionResult> executeTransaction(List<Command> commands, String path, long partitionId, long gen) {
        return nodeResolver.getNodeIdByPathAndPartitionIdOrRandom(path, partitionId)
                .executeTransaction(commands, path, partitionId, gen)
                .thenApply(r -> {
                    if (StatusCode.PRECONDITION_FAILED.equals(r.getCode())) {
                        throw new KikimrKvGenerationChangedRuntimeException(partitionId);
                    }
                    ExecuteTransactionResult result = r.expect("executeTransaction");
                    if (result.getNodeId() != 0) {
                        nodeResolver.updateNodeId(path, partitionId, result.getNodeId());
                    }
                    return result;
                });
    }

    public CompletableFuture<ReadRangeResult> readRange(String path, long partitionId, KeyRange keyRange, long generation, long limitBytes, Priority priority) {
        return nodeResolver.getNodeIdByPathAndPartitionIdOrRandom(path, partitionId)
                .readRange(path, partitionId, keyRange, generation, limitBytes, priority)
                .thenApply(r -> {
                    if (StatusCode.PRECONDITION_FAILED.equals(r.getCode())) {
                        throw new KikimrKvGenerationChangedRuntimeException(partitionId);
                    }
                    ReadRangeResult result = r.expect("readRange");
                    if (result.getNodeId() != 0) {
                        nodeResolver.updateNodeId(path, partitionId, result.getNodeId());
                    }
                    return result;
                });
    }

    public CompletableFuture<ListRangeResult> listRange(String path, long partitionId, KeyRange range, long generation, long limitBytes) {
        return nodeResolver.getNodeIdByPathAndPartitionIdOrRandom(path, partitionId)
                .listRange(path, partitionId, range, generation, limitBytes)
                .thenApply(r -> {
                    if (StatusCode.PRECONDITION_FAILED.equals(r.getCode())) {
                        throw new KikimrKvGenerationChangedRuntimeException(partitionId);
                    }
                    ListRangeResult result = r.expect("listRange");
                    if (result.getNodeId() != 0) {
                        nodeResolver.updateNodeId(path, partitionId, result.getNodeId());
                    }
                    return result;
                });
    }

    @Override
    public void close() {
        nodeResolver.close();
    }
}
