package ru.yandex.kikimr.client.kv;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.CheckReturnValue;

/**
 * @author Stepan Koltsov
 */
public interface KvAsyncIterator<A> {
    @CheckReturnValue
    CompletableFuture<Optional<A>> next();
    void commit();

    default <B> KvAsyncIterator<B> map(Function<A, B> f) {
        return new Map<>(this, f);
    }

    @CheckReturnValue
    default CompletableFuture<Void> readRemaining(Consumer<A> listener) {
        return next().thenCompose(o -> {
            if (o.isPresent()) {
                listener.accept(o.get());
                commit();
                return readRemaining(listener);
            } else {
                return CompletableFuture.completedFuture(null);
            }
        });
    }

    class Map<A, B> implements KvAsyncIterator<B> {
        private final KvAsyncIterator<A> iter;
        private final Function<A, B> f;

        public Map(KvAsyncIterator<A> iter, Function<A, B> f) {
            this.iter = iter;
            this.f = f;
        }

        @Override
        public CompletableFuture<Optional<B>> next() {
            return iter.next().thenApply(o -> o.map(f));
        }

        @Override
        public void commit() {
            iter.commit();
        }
    }
}
