package ru.yandex.kikimr.client.kv;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;

import ru.yandex.kikimr.util.NameRange;

/**
 * @author Stepan Koltsov
 */
public class KvAsyncIteratorReadRange implements KvAsyncIterator<ArrayList<KikimrKvClient.KvEntryWithStats>> {
    private final KikimrKvClient kikimrKvClient;
    private final long tabletId;
    private final long gen;
    @Nonnull
    private final NameRange nameRange;
    private final long limitBytes;

    public KvAsyncIteratorReadRange(
        KikimrKvClient kikimrKvClient,
        long tabletId, long gen, @Nonnull NameRange nameRange, long limitBytes)
    {

        this.kikimrKvClient = kikimrKvClient;
        this.tabletId = tabletId;
        this.gen = gen;
        this.nameRange = nameRange;
        this.limitBytes = limitBytes;
        this.state = new GoingToRead(nameRange);
    }

    private abstract class State {
        abstract CompletableFuture<Optional<ArrayList<KikimrKvClient.KvEntryWithStats>>> next();
        abstract void commit();
    }

    private class Done extends State {
        @Override
        CompletableFuture<Optional<ArrayList<KikimrKvClient.KvEntryWithStats>>> next() {
            return CompletableFuture.completedFuture(Optional.empty());
        }

        @Override
        void commit() {
        }
    }

    @Nonnull
    private State state;

    private class GoingToRead extends State {
        private final NameRange remainingRange;

        private GoingToRead(NameRange remainingRange) {
            this.remainingRange = remainingRange;
        }

        @Override
        CompletableFuture<Optional<ArrayList<KikimrKvClient.KvEntryWithStats>>> next() {
            return kikimrKvClient.readRange(tabletId, gen, remainingRange, true, limitBytes, 0)
                .thenApply(r -> {
                    WaitingForCommit newState = new WaitingForCommit(r);
                    state = newState;
                    return newState.makeNext();
                });
        }

        @Override
        void commit() {
            throw new IllegalStateException("must not call commit before next");
        }
    }

    private class WaitingForCommit extends State {
        @Nonnull
        private final KvReadRangeResult result;

        public WaitingForCommit(@Nonnull KvReadRangeResult result) {
            this.result = result;
        }

        private Optional<ArrayList<KikimrKvClient.KvEntryWithStats>> makeNext() {
            KikimrKvClient.KvEntryWithStats[] entries = result.getEntries();
            if (entries.length == 0) {
                return Optional.empty();
            } else {
                return Optional.of(new ArrayList<>(Arrays.asList(entries)));
            }
        }

        @Override
        CompletableFuture<Optional<ArrayList<KikimrKvClient.KvEntryWithStats>>> next() {
            return CompletableFuture.completedFuture(makeNext());
        }

        @Override
        void commit() {
            if (result.isLast()) {
                state = new Done();
            } else {
                String overrunLastEntryName = result.getOverrunLastEntryName();

                NameRange moreRange = NameRange.rangeFrom(overrunLastEntryName, false);
                if (nameRange.getEnd() != null) {
                    moreRange = moreRange.to(nameRange.getEnd(), nameRange.isEndInclusive());
                }

                state = new GoingToRead(moreRange);
            }

            // release memory if someone is holding a reference to result
            result.takeEntries();
        }
    }

    @Override
    public CompletableFuture<Optional<ArrayList<KikimrKvClient.KvEntryWithStats>>> next() {
        return state.next();
    }

    @Override
    public void commit() {
        state.commit();
    }
}
