package ru.yandex.kikimr.client.kv;

import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.ByteString;

import ru.yandex.kikimr.proto.Base;
import ru.yandex.kikimr.proto.MsgbusKv;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class KvCommandReadOne extends KvCommandTyped<MsgbusKv.TKeyValueRequest.TCmdRead, MsgbusKv.TKeyValueResponse.TReadResult> {

    @Nonnull
    private final String key;
    private final long offset;
    private final long length;
    private final MsgbusKv.TKeyValueRequest.EPriority priority;

    public KvCommandReadOne(@Nonnull String key, long offset, long length, MsgbusKv.TKeyValueRequest.EPriority priority) {
        this.key = key;
        this.offset = offset;
        this.length = length;
        this.priority = priority;

        if (length == 0) {
            throw new IllegalArgumentException("zero length read is pointless");
        }
    }

    @Override
    public KvCommandType<MsgbusKv.TKeyValueRequest.TCmdRead, MsgbusKv.TKeyValueResponse.TReadResult> type() {
        return KvCommandType.read;
    }

    @Override
    protected MsgbusKv.TKeyValueRequest.TCmdRead makeQ() {
        MsgbusKv.TKeyValueRequest.TCmdRead.Builder r = MsgbusKv.TKeyValueRequest.TCmdRead.newBuilder();
        r.setKey(ByteString.copyFromUtf8(key));
        r.setPriority(priority);
        if (length >= 0) {
            r.setOffset(offset);
            r.setSize(length);
        }
        return r.build();
    }

    @Override
    protected void updateCounters(MsgbusKv.TKeyValueResponse.TReadResult response, KvCounters counters) {
        counters.readCompleted(1, response.getValue().size());
    }

    @Nonnull
    public String getKey() {
        return key;
    }

    public Optional<byte[]> getData() {
        if (getResponse().getStatus() == Base.EReplyStatus.NODATA_VALUE) {
            return Optional.empty();
        } else {
            if (getResponse().getStatus() != Base.EReplyStatus.OK_VALUE) {
                // status must be checked earlier
                throw new IllegalStateException();
            }
            return Optional.of(getResponse().getValue().toByteArray());
        }
    }
}
