package ru.yandex.kikimr.client.kv;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.ByteString;

import ru.yandex.kikimr.client.ReplyStatus;
import ru.yandex.kikimr.proto.Base;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class KvCommandReadRange extends KvCommandTyped<MsgbusKv.TKeyValueRequest.TCmdReadRange, MsgbusKv.TKeyValueResponse.TReadRangeResult> {

    private final NameRange nameRange;
    private final boolean includeData;
    private final long limitBytes;

    public KvCommandReadRange(NameRange nameRange, boolean includeData, long limitBytes) {
        this.nameRange = nameRange;
        this.includeData = includeData;
        this.limitBytes = limitBytes;

        if (limitBytes == 0) {
            throw new IllegalArgumentException();
        }
    }

    @Override
    public KvCommandType<MsgbusKv.TKeyValueRequest.TCmdReadRange, MsgbusKv.TKeyValueResponse.TReadRangeResult> type() {
        return KvCommandType.readRange;
    }

    @Override
    protected MsgbusKv.TKeyValueRequest.TCmdReadRange makeQ() {
        MsgbusKv.TKeyValueRequest.TCmdReadRange.Builder r = MsgbusKv.TKeyValueRequest.TCmdReadRange.newBuilder();
        r.setRange(toKvRange(nameRange));
        r.setIncludeData(includeData);
        if (limitBytes > 0) {
            r.setLimitBytes(limitBytes);
        }
        return r.build();
    }

    @Override
    protected void checkStatus(@Nonnull MsgbusKv.TKeyValueResponse.TReadRangeResult response) {
        int status = response.getStatus();
        if (status == Base.EReplyStatus.OVERRUN_VALUE) {
            return;
        }
        ReplyStatus.checkOk(status, makeQ(), response);
    }

    @Override
    protected void updateCounters(MsgbusKv.TKeyValueResponse.TReadRangeResult response, KvCounters counters) {
        counters.readCompleted(
            response.getPairCount(),
            response.getPairList().stream().mapToLong(MsgbusKv.TKeyValueResponse.TKeyValuePair::getValueSize).sum());
    }

    private KikimrKvClient.KvEntryWithStats[] getEntries() {
        return getResponse().getPairList().stream()
            .map(p -> {
                return new KikimrKvClient.KvEntryWithStats(
                    p.getKey().toStringUtf8(),
                    p.getValue().toByteArray(),
                    p.getValueSize(),
                    p.getCreationUnixTime());
            })
            .toArray(KikimrKvClient.KvEntryWithStats[]::new);
    }

    public KvReadRangeResult getResult() {
        try {
            return new KvReadRangeResult(getEntries(), getResponse().getStatus() == Base.EReplyStatus.OVERRUN_VALUE);
        } catch (Exception e) {
            throw new RuntimeException("failed to execute: " + this, e);
        }
    }

    @Nonnull
    public static MsgbusKv.TKeyValueRequest.TKeyRange toKvRange(NameRange nameRange) {
        MsgbusKv.TKeyValueRequest.TKeyRange.Builder r = MsgbusKv.TKeyValueRequest.TKeyRange.newBuilder();

        if (nameRange.getBegin() != null) {
            r.setFrom(ByteString.copyFromUtf8(nameRange.getBegin()));
            r.setIncludeFrom(nameRange.isBeginInclusive());
        } else {
            r.setFrom(ByteString.EMPTY);
            r.setIncludeFrom(true);
        }

        if (nameRange.getEnd() != null) {
            r.setTo(ByteString.copyFromUtf8(nameRange.getEnd()));
            r.setIncludeTo(nameRange.isEndInclusive());
        } else {
            r.setTo(ByteString.copyFromUtf8("~"));
            r.setIncludeTo(true);
        }

        return r.build();
    }

    @Override
    public String toString() {
        return "KvCommandReadRange{" +
            "nameRange=" + nameRange +
            ", includeData=" + includeData +
            ", limitBytes=" + limitBytes +
            '}';
    }
}
