package ru.yandex.kikimr.client.kv;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;

import ru.yandex.kikimr.proto.MsgbusKv;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class KvCommandType<Q extends Message, A extends Message> {
    private final Q requestTemplate;
    private final A responseTemplate;
    private final Descriptors.FieldDescriptor requestField;
    private final Descriptors.FieldDescriptor responseField;
    private final Descriptors.FieldDescriptor responseStatusField;
    private final Type type;

    public KvCommandType(Q requestTemplate, A responseTemplate, Type type) {
        this.requestTemplate = requestTemplate;
        this.responseTemplate = responseTemplate;
        this.type = type;

        requestField = repeatedFieldByType(requestTemplate, MsgbusKv.TKeyValueRequest.getDefaultInstance());
        responseField = repeatedFieldByType(responseTemplate, MsgbusKv.TKeyValueResponse.getDefaultInstance());

        responseStatusField = responseTemplate.getDescriptorForType().findFieldByName("Status");
        if (responseStatusField.isRepeated()) {
            throw new IllegalStateException();
        }
        if (responseStatusField.getJavaType() != Descriptors.FieldDescriptor.JavaType.INT) {
            throw new IllegalStateException();
        }
    }

    private static <M extends Message> Descriptors.FieldDescriptor repeatedFieldByType(M requestTemplate, Message messageTemplate) {
        Descriptors.FieldDescriptor requestField = messageTemplate.getDescriptorForType().getFields()
            .stream()
            .filter(f -> {
                return f.getType() == Descriptors.FieldDescriptor.Type.MESSAGE
                    && f.getMessageType() == requestTemplate.getDescriptorForType();
            })
            .collect(Collectors.toList()).get(0);

        if (!requestField.isRepeated()) {
            throw new IllegalStateException();
        }

        return requestField;
    }

    @Nonnull
    public Type getType() {
        return type;
    }

    public int addRequest(MsgbusKv.TKeyValueRequest.Builder request, Q command) {
        int index = request.getRepeatedFieldCount(requestField);
        request.addRepeatedField(requestField, command);
        return index;
    }

    public void setRequests(MsgbusKv.TKeyValueRequest.Builder request, List<Q> commands) {
        request.setField(requestField, commands);
    }

    @Nonnull
    public List<A> getResponses(MsgbusKv.TKeyValueResponse response) {
        return (List<A>) response.getField(responseField);
    }

    @Nonnull
    public A getResponse(MsgbusKv.TKeyValueResponse response, int index) {
        return (A) response.getRepeatedField(responseField, index);
    }

    public int getResponseStatus(A commandResponse) {
        return (int) commandResponse.getField(responseStatusField);
    }

    public enum Type {
        DELETE_RANGE,
        READ,
        READ_RANGE,
        WRITE,
        RENAME,
        CLONE,
        CONCAT,
        ;
    }


    public static final KvCommandType<
        MsgbusKv.TKeyValueRequest.TCmdDeleteRange,
        MsgbusKv.TKeyValueResponse.TDeleteRangeResult> deleteRange =
            new KvCommandType<>(
                MsgbusKv.TKeyValueRequest.TCmdDeleteRange.getDefaultInstance(),
                MsgbusKv.TKeyValueResponse.TDeleteRangeResult.getDefaultInstance(),
                Type.DELETE_RANGE);

    public static final KvCommandType<
        MsgbusKv.TKeyValueRequest.TCmdRead,
        MsgbusKv.TKeyValueResponse.TReadResult> read =
            new KvCommandType<>(
                MsgbusKv.TKeyValueRequest.TCmdRead.getDefaultInstance(),
                MsgbusKv.TKeyValueResponse.TReadResult.getDefaultInstance(),
                Type.READ);

    public static final KvCommandType<
        MsgbusKv.TKeyValueRequest.TCmdReadRange,
        MsgbusKv.TKeyValueResponse.TReadRangeResult> readRange =
            new KvCommandType<>(
                MsgbusKv.TKeyValueRequest.TCmdReadRange.getDefaultInstance(),
                MsgbusKv.TKeyValueResponse.TReadRangeResult.getDefaultInstance(),
                Type.READ_RANGE);

    public static final KvCommandType<
        MsgbusKv.TKeyValueRequest.TCmdWrite,
        MsgbusKv.TKeyValueResponse.TWriteResult> write =
            new KvCommandType<>(
                MsgbusKv.TKeyValueRequest.TCmdWrite.getDefaultInstance(),
                MsgbusKv.TKeyValueResponse.TWriteResult.getDefaultInstance(),
                Type.WRITE);

    public static final KvCommandType<
        MsgbusKv.TKeyValueRequest.TCmdRename,
        MsgbusKv.TKeyValueResponse.TRenameResult> rename =
            new KvCommandType<>(
                MsgbusKv.TKeyValueRequest.TCmdRename.getDefaultInstance(),
                MsgbusKv.TKeyValueResponse.TRenameResult.getDefaultInstance(),
                Type.RENAME);

    public static final KvCommandType<
        MsgbusKv.TKeyValueRequest.TCmdCopyRange,
        MsgbusKv.TKeyValueResponse.TCopyRangeResult> clone =
            new KvCommandType<>(
                MsgbusKv.TKeyValueRequest.TCmdCopyRange.getDefaultInstance(),
                MsgbusKv.TKeyValueResponse.TCopyRangeResult.getDefaultInstance(),
                Type.CLONE);

    public static final KvCommandType<
        MsgbusKv.TKeyValueRequest.TCmdConcat,
        MsgbusKv.TKeyValueResponse.TConcatResult> concat =
            new KvCommandType<>(
                MsgbusKv.TKeyValueRequest.TCmdConcat.getDefaultInstance(),
                MsgbusKv.TKeyValueResponse.TConcatResult.getDefaultInstance(),
                Type.CONCAT);

}
