package ru.yandex.kikimr.client.kv;

import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;

import com.yandex.ydb.yndx.rate_limiter.ListRangeResult;
import com.yandex.ydb.yndx.rate_limiter.ReadRangeResult;

/**
 * @author Stepan Koltsov
 */
public class KvReadRangeResult {
    private KikimrKvClient.KvEntryWithStats[] entries;
    private final boolean overrun;

    public KvReadRangeResult(@Nonnull KikimrKvClient.KvEntryWithStats[] entries, boolean overrun) {
        if (entries == null) {
            throw new IllegalArgumentException();
        }

        this.entries = entries;
        this.overrun = overrun;

        if (overrun) {
            if (entries.length == 0) {
                throw new IllegalArgumentException(
                        "No entries and overrun. Likely limit exceeded");
            }
        }
    }

    public static KvReadRangeResult fromReadRange(@Nonnull List<ReadRangeResult.KeyValuePair> entries, boolean overrun) {
        return new KvReadRangeResult(entries.stream()
                .map(kv ->
                        new KikimrKvClient.KvEntryWithStats(
                                kv.getKey(),
                                kv.getValue().toByteArray(),
                                kv.getSerializedSize(),
                                kv.getCreationUnixTime()))
                .toArray(KikimrKvClient.KvEntryWithStats[]::new), overrun);
    }

    public static KvReadRangeResult fromListRange(@Nonnull List<ListRangeResult.KeyInfo> entries, boolean overrun) {
        return new KvReadRangeResult(entries.stream()
                .map(kv ->
                        new KikimrKvClient.KvEntryWithStats(
                                kv.getKey(),
                                new byte[0],
                                kv.getSerializedSize(),
                                kv.getCreationUnixTime()))
                .toArray(KikimrKvClient.KvEntryWithStats[]::new), overrun);
    }


    private void check() {
        if (entries == null) {
            throw new IllegalStateException();
        }
    }

    @Nonnull
    public KikimrKvClient.KvEntryWithStats[] getEntries() {
        check();
        return entries;
    }

    public KikimrKvClient.KvEntryWithStats[] takeEntries() {
        check();
        KikimrKvClient.KvEntryWithStats[] r = this.entries;
        entries = null;
        return r;
    }

    @Nonnull
    public String getOverrunLastEntryName() {
        if (!overrun) {
            throw new IllegalStateException();
        }
        return entries[entries.length - 1].getName();
    }

    public boolean isOverrun() {
        return overrun;
    }

    public boolean isLast() {
        return entries.length == 0 || !overrun;
    }

    public long bytesSize() {
        return Arrays.stream(entries).mapToLong(KikimrKvClient.KvEntryWithStats::getSize).sum();
    }

    @Nonnull
    public KikimrKvClient.KvEntryWithStats[] getEntriesAll() {
        if (overrun) {
            throw new IllegalStateException();
        }
        return getEntries();
    }
}
