package ru.yandex.kikimr.client.kv.inMem;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Optional;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvGenerationChangedRuntimeException;
import ru.yandex.kikimr.client.kv.KvReadRangeResult;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.concurrent.CompletableFutures;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class KikimrKvClientInMem implements KikimrKvClient {
    private int doNotExceedFileSize = DO_NOT_EXCEED_FILE_SIZE;

    private final ReentrantLock lock = new ReentrantLock();
    private final Map<String, Map<Long, Tablet>> tabletsByPath = new HashMap<>();
    private final HashMap<Long, Tablet> allTablets = new HashMap<>();
    private final AtomicLong tabletIdSeq = new AtomicLong(1);

    private static class Tablet {
        private long generation = 0;

        private final TreeMap<String, byte[]> files = new TreeMap<>();

        public long incrementGeneration() {
            return ++generation;
        }

        public void write(String key, byte[] value) {
            System.out.println("write " + key);
            files.put(key, value.clone());
        }

        private NavigableMap<String, byte[]> subMap(NameRange nameRange) {
            NavigableMap<String, byte[]> r = files;

            if (nameRange.getBegin() != null) {
                r = r.tailMap(nameRange.getBegin(), nameRange.isBeginInclusive());
            }
            if (nameRange.getEnd() != null) {
                r = r.headMap(nameRange.getEnd(), nameRange.isEndInclusive());
            }

            return r;
        }

        public void deleteRange(NameRange nameRange) {
            System.out.println("delete " + nameRange);
            subMap(nameRange).clear();
        }

        public void deleteRanges(List<NameRange> nameRanges) {
            for (NameRange nameRange : nameRanges) {
                deleteRange(nameRange);
            }
        }

        public KvReadRangeResult readRange(NameRange nameRange, boolean includeData, long limitBytes) {
            System.out.println("read range " + nameRange + " " + includeData);

            boolean overrun = false;

            ArrayList<KvEntryWithStats> r = new ArrayList<>();
            for (Map.Entry<String, byte[]> e : subMap(nameRange).entrySet()) {
                long currentSize = r.stream().mapToLong(v -> v.getValue().length).sum();
                if (limitBytes >= 0 && r.size() > 0 && currentSize + currentSize > limitBytes) {
                    overrun = true;
                    break;
                }

                // TODO: use actual created time
                long createdUnixtime = System.currentTimeMillis() / 1000;
                KvEntryWithStats x;
                if (includeData) {
                    x = new KvEntryWithStats(e.getKey(), e.getValue().clone(), e.getValue().length, createdUnixtime);
                } else {
                    x = new KvEntryWithStats(e.getKey(), new byte[0], e.getValue().length, createdUnixtime);
                }
                r.add(x);
            }
            KvEntryWithStats[] array = r.toArray(new KvEntryWithStats[0]);
            return new KvReadRangeResult(array, overrun);
        }

        public Optional<byte[]> readData(String name, long offset, long length) {
            System.out.println("read data " + name + "(offset: "+offset + ", length: "+length + ")");
            byte[] bytes = files.get(name);
            if (bytes == null) {
                return Optional.empty();
            }
            if (length < 0) {
                return Optional.of(Arrays.copyOfRange(bytes, (int) offset, bytes.length));
            } else {
                return Optional.of(Arrays.copyOfRange(bytes, (int) offset, Math.toIntExact(Math.min(offset + length, bytes.length))));
            }
        }

        public void rename(String from, String to) {
            System.out.println("rename " + from + " " + to);
            byte[] value = files.remove(from);
            if (value == null) {
                throw new IllegalStateException("key not found: " + from);
            }
            files.put(to, value);
        }

        public void cloneRange(NameRange nameRange, String addPrefix, String removePrefix) {
            ArrayList<KeyValue> add = new ArrayList<>();
            for (Map.Entry<String, byte[]> e : subMap(nameRange).entrySet()) {
                if (e.getKey().startsWith(removePrefix)) {
                    add.add(new KeyValue(addPrefix + e.getKey().substring(removePrefix.length()), e.getValue()));
                }
            }

            for (KeyValue e : add) {
                files.put(e.key, e.value);
            }
        }

        public void concatAndDeleteOriginals(List<String> inputs, String output) {
            System.out.println("concat " + inputs + " to " + output);
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            for (String input : inputs) {
                byte[] file = files.remove(input);
                if (file == null) {
                    throw new IllegalStateException();
                }

                try {
                    os.write(file);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }

            files.put(output, os.toByteArray());
        }
    }

    private <A> CompletableFuture<A> op(long tabletId, Function<Tablet, A> op) {
        lock.lock();
        try {
            Tablet tablet = allTablets.get(tabletId);

            if (tablet == null) {
                return CompletableFuture.failedFuture(
                    new RuntimeException("tablet not found by id: " + tabletId));
            }

            return CompletableFuture.completedFuture(op.apply(tablet));
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        } finally {
            lock.unlock();
        }
    }

    private <A> CompletableFuture<A> opWithGen(long tabletId, long gen, Function<Tablet, A> op) {
        return op(tabletId, tablet -> {
            if (tablet.generation != gen && gen != 0) {
                throw new KikimrKvGenerationChangedRuntimeException(tabletId);
            }
            return op.apply(tablet);
        });
    }

    public long createKvTablet() {
        long id = tabletIdSeq.getAndIncrement();
        createKvTabletById(UUID.randomUUID().toString(), id);
        return id;
    }

    public void createKvTabletById(String path, long id) {
        lock.lock();
        try {
            Tablet tablet = new Tablet();
            allTablets.putIfAbsent(id, tablet);
            tabletsByPath.computeIfAbsent(path, p -> new LinkedHashMap<>())
                .putIfAbsent(id, tablet);
        } finally {
            lock.unlock();
        }
    }

    public long getGeneration(long id) {
        lock.lock();
        try {
            return allTablets.get(id).generation;
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void setDoNotExceedFileSizeForTest(int doNotExceedFileSize) {
        this.doNotExceedFileSize = doNotExceedFileSize;
    }

    @Override
    public int getDoNotExceedFileSize() {
        return doNotExceedFileSize;
    }

    @Override
    public void close() {
        clear();
    }

    public void clear() {
        lock.lock();
        try {
            allTablets.clear();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public CompletableFuture<Void> createKvTablets(String path, int count) {
        for (int i = 0; i < count; i++) {
            long id = tabletIdSeq.getAndIncrement();
            createKvTabletById(path, id);
        }
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> alterKvTablets(String path, int count) {
        lock.lock();
        try {
            Map<Long, Tablet> tablets = tabletsByPath.get(path);
            if (tablets == null) {
                return CompletableFuture.failedFuture(new IllegalStateException("KV volume not exists"));
            }

            if (tablets.size() > count) {
                return CompletableFuture.failedFuture(new IllegalStateException("Unable reduce count tablets"));
            }

            for (int i = tablets.size(); i < count; i++) {
                long id = tabletIdSeq.getAndIncrement();
                createKvTabletById(path, id);
            }
        } finally {
            lock.unlock();
        }
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> dropKvTablets(String path) {
        lock.lock();
        try {
            Map<Long, Tablet> byPath = tabletsByPath.remove(path);
            if (byPath != null) {
                for (Long tabletId : byPath.keySet()) {
                    allTablets.remove(tabletId);
                }
            }
        } finally {
            lock.unlock();
        }
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<long[]> resolveKvTablets(String path) {
        lock.lock();
        try {
            Map<Long, Tablet> byPath = tabletsByPath.get(path);
            if (byPath != null) {
                long[] tabletIds = byPath.keySet().stream()
                    .mapToLong(id -> id)
                    .toArray();
                return CompletableFuture.completedFuture(tabletIds);
            }
            return CompletableFuture.completedFuture(new long[0]);
        } finally {
            lock.unlock();
        }
    }

    @Override
    public CompletableFuture<long[]> findTabletsOnLocalhost() {
        lock.lock();
        try {
            long[] tabletIds = allTablets.keySet().stream()
                .mapToLong(id -> id)
                .toArray();
            return CompletableFuture.completedFuture(tabletIds);
        } finally {
            lock.unlock();
        }
    }

    @Override
    public CompletableFuture<Long> incrementGeneration(long tabletId, long expiredAt) {
        return op(tabletId, Tablet::incrementGeneration);
    }

    @Override
    public CompletableFuture<Void> writeAndRenameAndDeleteAndConcat(
        long tabletId, long gen,
        List<Write> writes,
        List<Rename> renames,
        List<NameRange> deletes,
        List<Concat> concats, long expiredAt)
    {
        return opWithGen(tabletId, gen, tablet -> {
            for (Write write : writes) {
                tablet.write(write.getName(), write.getValue().toByteArray());
            }
            for (Rename rename : renames) {
                tablet.rename(rename.getFrom(), rename.getTo());
            }
            for (NameRange delete : deletes) {
                tablet.deleteRange(delete);
            }
            for (Concat concat : concats) {
                if (concat.isKeepInputs()) {
                    throw new IllegalArgumentException("TODO");
                }
                tablet.concatAndDeleteOriginals(concat.getInputs(), concat.getOutput());
            }

            return null;
        });
    }

    @Override
    public CompletableFuture<Void> copyRange(long tabletId, long gen, NameRange nameRange, String addPrefix, String removePrefix, long expiredAt) {
        return opWithGen(tabletId, gen, tablet -> {
            tablet.cloneRange(nameRange, addPrefix, removePrefix);
            return null;
        });
    }

    @Override
    public CompletableFuture<KvReadRangeResult> readRange(long tabletId, long gen, NameRange nameRange, boolean includeData, long limitBytes, long expiredAt) {
        return runMaybeReadPaused(() -> {
            return opWithGen(tabletId, gen, tablet -> {
                return tablet.readRange(nameRange, includeData, limitBytes);
            });
        });
    }

    @Override
    public CompletableFuture<Optional<byte[]>> readData(long tabletId, long gen, String name, long offset, long length, long expiredAt, MsgbusKv.TKeyValueRequest.EPriority priority) {
        return runMaybeReadPaused(() -> {
            return opWithGen(tabletId, gen, tablet -> tablet.readData(name, offset, length));
        });
    }


    private <A> CompletableFuture<A> runPaused(Supplier<CompletableFuture<A>> op) {
        lock.lock();
        try {
            if (readMode != ReadMode.PAUSED) {
                return runMaybeReadPaused(op);
            }

            CompletableFuture<A> r = new CompletableFuture<>();
            pausedReads.add(() -> {
                CompletableFutures.whenComplete(op.get(), r);
            });
            return r;
        } finally {
            lock.unlock();
        }
    }

    private <A> CompletableFuture<A> runMaybeReadPaused(Supplier<CompletableFuture<A>> op) {
        switch (readMode) {
            case NORMAL:
                return op.get();
            case PAUSED:
                return runPaused(op);
            case THROW:
                throw new RuntimeException("readMode == " + ReadMode.THROW);
            default:
                throw new IllegalStateException();
        }
    }


    private ArrayList<Runnable> pausedReads = new ArrayList<>();

    private enum ReadMode {
        NORMAL,
        PAUSED,
        THROW,
    }

    private volatile ReadMode readMode = ReadMode.NORMAL;

    public void pauseReads() {
        readMode = ReadMode.PAUSED;
    }

    public void throwOnRead() {
        readMode = ReadMode.THROW;
    }

    public void resumeReads() {
        ArrayList<Runnable> pausedOps;

        lock.lock();
        try {
            readMode = ReadMode.NORMAL;

            pausedOps = this.pausedReads;
            this.pausedReads = new ArrayList<>();
        } finally {
            lock.unlock();
        }

        for (Runnable pausedOp : pausedOps) {
            pausedOp.run();
        }
    }

    private static class KeyValue {
        public final String key;
        public final byte[] value;

        public KeyValue(String key, byte[] value) {
            this.key = key;
            this.value = value;
        }
    }
}
