package ru.yandex.kikimr.client.kv.noderesolver;

import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import NKikimrClient.TGRpcServerGrpc;
import com.google.common.net.HostAndPort;
import io.netty.util.collection.LongObjectHashMap;
import io.netty.util.collection.LongObjectMap;

import ru.yandex.kikimr.client.KikimrAnyResponseException;
import ru.yandex.kikimr.client.ResponseStatus;
import ru.yandex.kikimr.grpc.GrpcTransport;
import ru.yandex.kikimr.proto.Msgbus;
import ru.yandex.kikimr.proto.Tablet;
import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.util.actors.PingActorRunner;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class KikimrKvNodeResolverImpl implements KikimrKvNodeResolver, AutoCloseable {
    private static final long REFRESH_INTERVAL_MILLIS = TimeUnit.SECONDS.toMillis(5);
    private static final long SNAPSHOT_MAX_TIME_TO_LIVE_NANOS = TimeUnit.SECONDS.toNanos(60);
    private static final long REQUEST_TIMEOUT = TimeUnit.SECONDS.toMillis(10);

    private final GrpcTransport transport;
    private final Executor executor;
    private final ScheduledExecutorService timer;
    private final ActorRunner parentActor;

    private final ConcurrentMap<HostAndPort, Node> nodeByAddress = new ConcurrentHashMap<>();
    private volatile LongObjectMap<HostAndPort> addressByTabletId = new LongObjectHashMap<>();
    private CompletionStage<Void> onChange;
    private volatile boolean closed;

    public KikimrKvNodeResolverImpl(GrpcTransport transport, Executor executor, ScheduledExecutorService timer) {
        this.transport = transport;
        this.executor = executor;
        this.timer = timer;
        this.parentActor = new ActorRunner(this::act, executor);
        this.parentActor.schedule();
    }

    @Nullable
    @Override
    public HostAndPort resolveByTabletId(long tabletId) {
        return addressByTabletId.get(tabletId);
    }

    private void act() {
        if (closed) {
            nodeByAddress.values().forEach(Node::close);
            nodeByAddress.clear();
            addressByTabletId = new LongObjectHashMap<>();
            return;
        }

        subscribeOnChangeIfNecessary();
        actualizeNodes();
        refreshTabletToAddress();
    }

    private void subscribeOnChangeIfNecessary() {
        var actualOnChange = transport.discovery().onChange();
        if (onChange != actualOnChange) {
            onChange = actualOnChange;
            onChange.whenComplete((ignore, e) -> parentActor.schedule());
        }
    }

    private void actualizeNodes() {
        var addresses = transport.discovery().addresses();
        removeOld(addresses);
        addNew(addresses);
    }

    private void removeOld(Set<HostAndPort> addresses) {
        var it = nodeByAddress.values().iterator();
        while (it.hasNext()) {
            var node = it.next();
            if (addresses.contains(node.address)) {
                continue;
            }

            node.close();
            it.remove();
        }
    }

    private void addNew(Set<HostAndPort> addresses) {
        for (var address : addresses) {
            if (nodeByAddress.containsKey(address)) {
                continue;
            }

            var node = new Node(address);
            nodeByAddress.put(address, node);
            node.schedule();
        }
    }

    private void refreshTabletToAddress() {
        List<Snapshot> snapshots = nodeByAddress.values().stream()
                .map(node -> node.snapshot)
                .sorted()
                .collect(toList());

        LongObjectMap<HostAndPort> result = new LongObjectHashMap<>(addressByTabletId.size());
        for (Snapshot snapshot : snapshots) {
            for (long tabletId : snapshot.tabletIds) {
                result.put(tabletId, snapshot.address);
            }
        }
        this.addressByTabletId = result;
    }

    @Override
    public void close() {
        closed = true;
        parentActor.schedule();
    }

    private class Node implements AutoCloseable {
        private final HostAndPort address;
        private final PingActorRunner actor;
        private volatile Snapshot snapshot;

        public Node(HostAndPort address) {
            this.address = address;
            this.snapshot = new Snapshot(address, new long[0]);
            this.actor = PingActorRunner.newBuilder()
                    .operation("resolve_local_tablets_on_node_" + address)
                    .onPing(this::act)
                    .executor(executor)
                    .timer(timer)
                    .pingInterval(Duration.ofMillis(REFRESH_INTERVAL_MILLIS))
                    .backoffDelay(Duration.ofSeconds(1))
                    .backoffMaxDelay(Duration.ofMinutes(5))
                    .build();
        }

        private void schedule() {
            actor.schedule();
        }

        private CompletableFuture<Void> act(int attempt) {
            return resolveLocalTablets().thenAccept(response -> {
                long[] tabletIds = response.getTabletInfoList()
                        .stream()
                        .mapToLong(Msgbus.TResponse.TTabletInfo::getTabletId)
                        .toArray();

                updateSnapshot(new Snapshot(address, tabletIds));
            }).whenComplete((ignore, e) -> {
                if (e != null && snapshot.ageNanos() > SNAPSHOT_MAX_TIME_TO_LIVE_NANOS) {
                    updateSnapshot(new Snapshot(address, new long[0]));
                }
            });
        }

        private void updateSnapshot(Snapshot update) {
            Snapshot prev = snapshot;
            snapshot = update;
            if (!snapshot.equals(prev)) {
                parentActor.schedule();
            }
        }

        private CompletableFuture<Msgbus.TResponse> resolveLocalTablets() {
            var request = Msgbus.TLocalEnumerateTablets.newBuilder()
                    .setDomainUid(1)
                    .setTabletType(Tablet.TTabletTypes.EType.KeyValue)
                    .build();

            long expiredAt = System.currentTimeMillis() + REQUEST_TIMEOUT;
            return transport.unaryCall(TGRpcServerGrpc.getLocalEnumerateTabletsMethod(), request, expiredAt, address)
                    .thenApply(response -> {
                        if (!ResponseStatus.isSuccess(response.getStatus(), false)) {
                            String op = Msgbus.TLocalEnumerateTablets.getDescriptor().getName();
                            throw new KikimrAnyResponseException(response, op, response.getErrorReason());
                        }
                        return response;
                    });
        }

        @Override
        public void close() {
            actor.close();
        }

        @Override
        public String toString() {
            return "Node{" +
                    "host='" + address + '\'' +
                    ", snapshot=" + snapshot +
                    '}';
        }
    }

    private static class Snapshot implements Comparable<Snapshot> {
        private final HostAndPort address;
        private final long[] tabletIds;
        private final long cratedAtNanos;

        public Snapshot(HostAndPort address, long[] tabletIds) {
            this.address = address;
            this.tabletIds = tabletIds;
            this.cratedAtNanos = System.nanoTime();
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (!(o instanceof Snapshot)) return false;

            Snapshot snapshot = (Snapshot) o;

            return Arrays.equals(tabletIds, snapshot.tabletIds);
        }

        public long ageNanos() {
            return System.nanoTime() - cratedAtNanos;
        }

        @Override
        public int hashCode() {
            return Arrays.hashCode(tabletIds);
        }

        @Override
        public int compareTo(Snapshot o) {
            return Long.compare(cratedAtNanos, o.cratedAtNanos);
        }

        @Override
        public String toString() {
            return "Snapshot{" +
                    "tabletIds=" + tabletIds.length +
                    '}';
        }
    }
}
