package ru.yandex.kikimr.client.kv.noderesolver;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.google.common.net.HostAndPort;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.yndx.rate_limiter.AcquireLockResult;
import com.yandex.ydb.yndx.rate_limiter.DescribeVolumeResult;
import com.yandex.ydb.yndx.rate_limiter.ExecuteTransactionResult;
import com.yandex.ydb.yndx.rate_limiter.ListLocalPartitionsResult;
import com.yandex.ydb.yndx.rate_limiter.ListRangeResult;
import com.yandex.ydb.yndx.rate_limiter.ReadRangeResult;
import com.yandex.ydb.yndx.rate_limiter.ReadResult;

import ru.yandex.kikimr.client.kv.commands.Command;
import ru.yandex.kikimr.client.kv.commands.KeyRange;
import ru.yandex.kikimr.client.kv.commands.Priority;

/**
 * @author senyasdr
 */
public interface Node extends AutoCloseable {

    int nodeId();

    HostAndPort hostAndPort();

    boolean isClosed();

    CompletableFuture<Result<ListLocalPartitionsResult>> listLocalPartitions(String path);

    /**
     * listLocalPartitions find all partitions of node
     *
     * @param nodeId ID of the node to get partitions for.
     *               If nodeId is equal to 0, then it is necessary to return the partitionIds of the node to which the request was sent
     * @param path   volume path
     * @return partition ids
     * @see <a href="https://a.yandex-team.ru/arcadia/kikimr/yndx/api/grpc/ydb_yndx_keyvalue_v1.proto?rev=r9386382#L24">ydb_yndx_keyvalue_v1.proto</a>
     */
    CompletableFuture<Result<ListLocalPartitionsResult>> listLocalPartitions(
            int nodeId, String path);

    CompletableFuture<Status> createKvVolume(String path, int count);

    CompletableFuture<Status> dropKvVolume(String path);

    CompletableFuture<Status> alterVolume(String path, int newPartitionCount);

    CompletableFuture<Result<DescribeVolumeResult>> describeVolume(String path);

    /**
     * Lock volume and increment generation
     *
     * @param partitionId partition id of volume
     * @param path        path af volume
     * @param expiredAt   nanos when request will expired
     * @return ID of new generation
     * @see <a href="https://a.yandex-team.ru/arcadia/kikimr/yndx/api/grpc/ydb_yndx_keyvalue_v1.proto?rev=r9386382#L27">ydb_yndx_keyvalue_v1.proto</a>
     */
    CompletableFuture<Result<AcquireLockResult>> acquireLock(long partitionId, String path, long expiredAt);

    CompletableFuture<Result<ReadResult>> read(String path, long partitionId, String key, long gen, long offset, long size, long limitBytes);

    CompletableFuture<Result<ExecuteTransactionResult>> executeTransaction(List<Command> commands, String path, long partitionId, long gen);

    CompletableFuture<Result<ReadRangeResult>> readRange(String path, long partitionId, KeyRange keyRange, long generation, long limitBytes, Priority priority);

    /**
     * Get list keys and metadata of items with keys in the specified range
     *
     * @param path        Volume path.
     * @param partitionId Partition of the volume.
     * @param range       The range of keys to read
     * @param generation  Generation of the exclusive lock acquired for the partition as a result of an AcquireLock call.
     * @param limitBytes  Result protobuf size limit. If not 0, overrides the default one only with a smaller value.
     * @return list keys and metadata
     * @see <a href=https://a.yandex-team.ru/arcadia/kikimr/yndx/api/grpc/ydb_yndx_keyvalue_v1.proto?rev=r9386382#L39>ydb_yndx_keyvalue_v1.proto</a>
     * @see <a href=https://a.yandex-team.ru/arcadia/kikimr/yndx/api/protos/ydb_yndx_keyvalue.proto?rev=r9386382#L368>ydb_yndx_keyvalue.proto</a>
     */
    CompletableFuture<Result<ListRangeResult>> listRange(String path, long partitionId, KeyRange range, long generation, long limitBytes);

    @Override
    void close();
}
