package ru.yandex.kikimr.client.kv.noderesolver;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import com.google.common.net.HostAndPort;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.grpc.GrpcRequestSettings;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.yndx.keyvalue.v1.KeyValueServiceGrpc;
import com.yandex.ydb.yndx.rate_limiter.AcquireLockRequest;
import com.yandex.ydb.yndx.rate_limiter.AcquireLockResult;
import com.yandex.ydb.yndx.rate_limiter.AlterVolumeRequest;
import com.yandex.ydb.yndx.rate_limiter.CreateVolumeRequest;
import com.yandex.ydb.yndx.rate_limiter.DescribeVolumeRequest;
import com.yandex.ydb.yndx.rate_limiter.DescribeVolumeResult;
import com.yandex.ydb.yndx.rate_limiter.DropVolumeRequest;
import com.yandex.ydb.yndx.rate_limiter.ExecuteTransactionRequest;
import com.yandex.ydb.yndx.rate_limiter.ExecuteTransactionResult;
import com.yandex.ydb.yndx.rate_limiter.ListLocalPartitionsRequest;
import com.yandex.ydb.yndx.rate_limiter.ListLocalPartitionsResult;
import com.yandex.ydb.yndx.rate_limiter.ListRangeRequest;
import com.yandex.ydb.yndx.rate_limiter.ListRangeResult;
import com.yandex.ydb.yndx.rate_limiter.ReadRangeRequest;
import com.yandex.ydb.yndx.rate_limiter.ReadRangeResult;
import com.yandex.ydb.yndx.rate_limiter.ReadRequest;
import com.yandex.ydb.yndx.rate_limiter.ReadResult;

import ru.yandex.kikimr.client.kv.commands.Command;
import ru.yandex.kikimr.client.kv.commands.KeyRange;
import ru.yandex.kikimr.client.kv.commands.Priority;

/**
 * @author senyasdr
 */
public class NodeImpl implements Node {

    private final int nodeId;
    private final HostAndPort hostAndPort;
    private final GrpcTransport transport;
    private volatile boolean isClosed = false;
    private static final int DEFAULT_NODE_ID = 0;

    public NodeImpl(int nodeId, HostAndPort hostAndPort, GrpcTransport transport) {
        this.nodeId = nodeId;
        this.hostAndPort = hostAndPort;
        this.transport = transport;
    }

    @Override
    public int nodeId() {
        return nodeId;
    }

    @Override
    public HostAndPort hostAndPort() {
        return hostAndPort;
    }

    @Override
    public boolean isClosed() {
        return isClosed;
    }

    @Override
    public CompletableFuture<Result<ListLocalPartitionsResult>> listLocalPartitions(String path) {
        return listLocalPartitions(DEFAULT_NODE_ID, path);
    }

    @Override
    public CompletableFuture<Result<ListLocalPartitionsResult>> listLocalPartitions(
            int nodeId, String path) {
        ListLocalPartitionsRequest req = ListLocalPartitionsRequest.newBuilder()
                .setNodeId(nodeId).setPath(path).build();
        final GrpcRequestSettings grpcRequestSettings = GrpcRequestSettings.newBuilder().build();
        return transport.unaryCall(KeyValueServiceGrpc.getListLocalPartitionsMethod(), req, grpcRequestSettings).thenCompose(r -> {
            if (!r.isSuccess()) {
                return CompletableFuture.completedFuture(r.cast());
            }
            return transport.getOperationTray().waitResult(
                    r.expect("listLocalPartitions()").getOperation(),
                    ListLocalPartitionsResult.class,
                    Function.identity(),
                    grpcRequestSettings);
        });
    }

    @Override
    public CompletableFuture<Status> createKvVolume(String path, int count) {
        CreateVolumeRequest req = CreateVolumeRequest.newBuilder()
                .setPath(path).setPartitionCount(count).build();
        final GrpcRequestSettings grpcRequestSettings = GrpcRequestSettings.newBuilder().build();
        return transport.unaryCall(KeyValueServiceGrpc.getCreateVolumeMethod(), req, grpcRequestSettings)
                .thenCompose(r -> {
                    if (!r.isSuccess()) {
                        return CompletableFuture.completedFuture(r.toStatus());
                    }
                    return transport.getOperationTray().waitStatus(
                            r.expect("createKvVolume()").getOperation(),
                            grpcRequestSettings);
                });
    }

    @Override
    public CompletableFuture<Status> dropKvVolume(String path) {
        DropVolumeRequest request = DropVolumeRequest.newBuilder().setPath(path).build();
        final GrpcRequestSettings grpcRequestSettings = GrpcRequestSettings.newBuilder().build();
        return transport.unaryCall(KeyValueServiceGrpc.getDropVolumeMethod(), request, grpcRequestSettings).thenCompose(r -> {
            if (!r.isSuccess()) {
                return CompletableFuture.completedFuture(r.toStatus());
            }
            return transport.getOperationTray().waitStatus(
                    r.expect("dropKvVolume()").getOperation(), grpcRequestSettings);
        });
    }

    @Override
    public CompletableFuture<Status> alterVolume(String path, int newPartitionCount) {
        AlterVolumeRequest request = AlterVolumeRequest.newBuilder().setPath(path)
                .setAlterPartitionCount(newPartitionCount).build();
        final GrpcRequestSettings grpcRequestSettings = GrpcRequestSettings.newBuilder().build();
        return transport.unaryCall(KeyValueServiceGrpc.getAlterVolumeMethod(), request, grpcRequestSettings).thenCompose(r -> {
            if (!r.isSuccess()) {
                return CompletableFuture.completedFuture(r.toStatus());
            }
            return transport.getOperationTray().waitStatus(
                    r.expect("alterVolume()").getOperation(), grpcRequestSettings);
        });
    }

    @Override
    public CompletableFuture<Result<DescribeVolumeResult>> describeVolume(String path) {
        DescribeVolumeRequest request = DescribeVolumeRequest.newBuilder().setPath(path).build();
        final GrpcRequestSettings grpcRequestSettings = GrpcRequestSettings.newBuilder().build();
        return transport.unaryCall(KeyValueServiceGrpc.getDescribeVolumeMethod(), request, grpcRequestSettings)
                .thenCompose(r -> {
                    if (!r.isSuccess()) {
                        return CompletableFuture.completedFuture(r.cast());
                    }
                    return transport.getOperationTray().waitResult(
                            r.expect("describeVolume()").getOperation(),
                            DescribeVolumeResult.class,
                            Function.identity(),
                            GrpcRequestSettings.newBuilder().build());
                });
    }

    @Override
    public CompletableFuture<Result<AcquireLockResult>> acquireLock(long partitionId, String path, long expiredAt) {
        AcquireLockRequest req = AcquireLockRequest.newBuilder().setPath(path).setPartitionId(partitionId).build();
        GrpcRequestSettings grpcRequestSettings = GrpcRequestSettings.newBuilder().withDeadlineAfter(expiredAt).build();
        return transport.unaryCall(KeyValueServiceGrpc.getAcquireLockMethod(), req, grpcRequestSettings)
                .thenCompose(r -> {
                    if (!r.isSuccess()) {
                        return CompletableFuture.completedFuture(r.cast());
                    }
                    return transport.getOperationTray().waitResult(
                            r.expect("acquireLock()").getOperation(),
                            AcquireLockResult.class,
                            Function.identity(),
                            GrpcRequestSettings.newBuilder().build());
                });
    }

    @Override
    public CompletableFuture<Result<ReadResult>> read(String path, long partitionId, String key, long gen, long offset, long size, long limitBytes) {
        ReadRequest.Builder requestBuilder = ReadRequest.newBuilder();

        requestBuilder.setPath(path).setPartitionId(partitionId).setKey(key).setOffset(offset).setSize(size).setLimitBytes(limitBytes);
        if (gen > 0) {
            requestBuilder.setLockGeneration(gen);
        }
        final GrpcRequestSettings grpcRequestSettings = GrpcRequestSettings.newBuilder().build();
        return transport.unaryCall(KeyValueServiceGrpc.getReadMethod(), requestBuilder.build(), grpcRequestSettings).thenCompose(r -> {
            if (!r.isSuccess()) {
                return CompletableFuture.completedFuture(r.cast());
            }
            return transport.getOperationTray().waitResult(
                    r.expect("read()").getOperation(),
                    ReadResult.class,
                    Function.identity(),
                    grpcRequestSettings);
        });
    }

    @Override
    public CompletableFuture<Result<ExecuteTransactionResult>> executeTransaction(List<Command> commands, String path, long partitionId, long gen) {
        ExecuteTransactionRequest.Builder reqBuilder = ExecuteTransactionRequest.newBuilder().setPath(path).setPartitionId(partitionId);
        for (Command command : commands) {
            reqBuilder.addCommands(command.toProtoCommand());
        }
        if (gen >= 0) {
            reqBuilder.setLockGeneration(gen);
        }
        final GrpcRequestSettings grpcRequestSettings = GrpcRequestSettings.newBuilder().build();
        return transport.unaryCall(KeyValueServiceGrpc.getExecuteTransactionMethod(), reqBuilder.build(), grpcRequestSettings).thenCompose(r -> {
            if (!r.isSuccess()) {
                return CompletableFuture.completedFuture(r.cast());
            }
            return transport.getOperationTray().waitResult(
                    r.expect("executeTransaction()").getOperation(),
                    ExecuteTransactionResult.class,
                    Function.identity(),
                    grpcRequestSettings);
        });
    }

    @Override
    public CompletableFuture<Result<ReadRangeResult>> readRange(String path, long partitionId, KeyRange keyRange, long generation, long limitBytes, Priority priority) {
        ReadRangeRequest.Builder requestBuilder = ReadRangeRequest.newBuilder();
        requestBuilder.setPath(path).setPartitionId(partitionId)
                .setLimitBytes(limitBytes).setPriority(priority.getProto())
                .setRange(keyRange.toProto());
        if (generation >= 0) {
            requestBuilder.setLockGeneration(generation);
        }

        final GrpcRequestSettings grpcRequestSettings = GrpcRequestSettings.newBuilder().build();
        return transport.unaryCall(KeyValueServiceGrpc.getReadRangeMethod(), requestBuilder.build(), grpcRequestSettings).thenCompose(r -> {
            if (!r.isSuccess()) {
                return CompletableFuture.completedFuture(r.cast());
            }
            return transport.getOperationTray().waitResult(
                    r.expect("readRange()").getOperation(),
                    ReadRangeResult.class,
                    Function.identity(),
                    grpcRequestSettings);
        });
    }

    @Override
    public CompletableFuture<Result<ListRangeResult>> listRange(String path, long partitionId, KeyRange range, long generation, long limitBytes) {
        ListRangeRequest.Builder requestBuilder = ListRangeRequest.newBuilder();
        requestBuilder.setPath(path).setPartitionId(partitionId).setRange(range.toProto()).setLimitBytes(limitBytes);
        if (generation >= 0) {
            requestBuilder.setLockGeneration(generation);
        }
        final GrpcRequestSettings grpcRequestSettings = GrpcRequestSettings.newBuilder().build();
        return transport.unaryCall(KeyValueServiceGrpc.getListRangeMethod(), requestBuilder.build(), grpcRequestSettings).thenCompose(r -> {
            if (!r.isSuccess()) {
                return CompletableFuture.completedFuture(r.cast());
            }
            return transport.getOperationTray().waitResult(
                    r.expect("listRange()").getOperation(),
                    ListRangeResult.class,
                    Function.identity(),
                    grpcRequestSettings);
        });
    }

    @Override
    public void close() {
        if (isClosed) {
            return;
        }
        synchronized (this) {
            if (isClosed) {
                return;
            }
            isClosed = true;
            if (transport != null) {
                transport.close();
            }
        }
    }
}
