package ru.yandex.kikimr.grpc;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.WillNotClose;

import com.google.common.collect.ImmutableMap;
import io.grpc.internal.GrpcUtil;
import io.grpc.netty.NettyChannelBuilder;
import io.netty.buffer.ByteBufAllocator;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcOptions {
    public static final Duration DEFAULT_CONNECT_TIMEOUT = Duration.ofSeconds(10);
    public static final Duration DEFAULT_READ_TIMEOUT = Duration.ofSeconds(30);

    public final Map<ChannelOption, Object> channelOptions;
    public final Consumer<NettyChannelBuilder> channelInitializer;
    public final int maxMessageSizeBytes;
    @WillNotClose
    @Nullable
    public final EventLoopGroup ioExecutor;
    @WillNotClose
    @Nullable
    public final Executor callExecutor;
    @WillNotClose
    @Nullable
    public final ScheduledExecutorService timer;
    public final long keepAliveNanos;
    public final long keepAliveTimeoutNanos;
    public final long readTimeoutMillis;
    public final long idleTimeoutMillis;
    public final int flowControlWindow;

    public GrpcOptions(Builder builder) {
        channelOptions = ImmutableMap.copyOf(builder.channelOptions);
        maxMessageSizeBytes = builder.maxMessageSizeBytes == 0
            ? 26 << 20
            : builder.maxMessageSizeBytes;
        ioExecutor = builder.ioExecutor;
        callExecutor = builder.callExecutor;
        timer = builder.timer;
        keepAliveNanos = builder.keepAliveNanos == 0
            ? GrpcUtil.DEFAULT_KEEPALIVE_TIMEOUT_NANOS
            : builder.keepAliveNanos;
        keepAliveTimeoutNanos = builder.keepAliveTimeoutNanos == 0
            ? GrpcUtil.DEFAULT_KEEPALIVE_TIMEOUT_NANOS
            : builder.keepAliveTimeoutNanos;
        readTimeoutMillis = builder.readTimeoutMillis == 0
            ? DEFAULT_READ_TIMEOUT.toMillis()
            : builder.readTimeoutMillis;
        idleTimeoutMillis = builder.idleTimeoutMillis == 0
            ? TimeUnit.MINUTES.toMillis(30) // default from gRPC
            : builder.idleTimeoutMillis;
        flowControlWindow = builder.flowControlWindow;
        channelInitializer = builder.channelInitializer;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private final Map<ChannelOption, Object> channelOptions = new HashMap<>();
        private int maxMessageSizeBytes;
        @WillNotClose
        @Nullable
        private EventLoopGroup ioExecutor;
        @WillNotClose
        @Nullable
        private Executor callExecutor;
        @WillNotClose
        @Nullable
        private ScheduledExecutorService timer;
        private Consumer<NettyChannelBuilder> channelInitializer = (cb) -> {};
        private long keepAliveNanos;
        private long keepAliveTimeoutNanos;
        private long readTimeoutMillis;
        private long idleTimeoutMillis;
        private int flowControlWindow;

        public Builder() {
            channelOptions.put(ChannelOption.ALLOCATOR, ByteBufAllocator.DEFAULT);
            channelOptions.put(ChannelOption.TCP_NODELAY, Boolean.TRUE);
            channelOptions.put(ChannelOption.CONNECT_TIMEOUT_MILLIS, Math.toIntExact(DEFAULT_CONNECT_TIMEOUT.toMillis()));
        }

        public Builder maxInboundMessageSize(int maxMessageSizeBytes) {
            this.maxMessageSizeBytes = maxMessageSizeBytes;
            return this;
        }

        public <T> Builder withOption(ChannelOption<T> option, T value) {
            channelOptions.put(option, value);
            return this;
        }

        public Builder eventLoopGroup(@WillNotClose EventLoopGroup ioExecutor) {
            this.ioExecutor = ioExecutor;
            return this;
        }

        public Builder callExecutor(@WillNotClose Executor callExecutor) {
            this.callExecutor = callExecutor;
            return this;
        }

        public Builder keepAliveTime(long keepAliveTime, TimeUnit timeUnit) {
            keepAliveNanos = timeUnit.toNanos(keepAliveTime);
            return this;
        }

        public Builder keepAliveTimeout(long keepAliveTimeout, TimeUnit timeUnit) {
            keepAliveTimeoutNanos = timeUnit.toNanos(keepAliveTimeout);
            return this;
        }

        public Builder readTimeout(long timeout, TimeUnit unit) {
            readTimeoutMillis = unit.toMillis(timeout);
            return this;
        }

        public Builder timer(ScheduledExecutorService timer) {
            this.timer = timer;
            return this;
        }

        public Builder channelInitializer(Consumer<NettyChannelBuilder> channelInitializer) {
            this.channelInitializer = checkNotNull(channelInitializer, "channelInitializer is null");
            return this;
        }

        public Builder idleTimeout(long timeout, TimeUnit unit) {
            this.idleTimeoutMillis = unit.toMillis(timeout);
            return this;
        }

        public Builder flowControlWindow(int flowControlWindow) {
            this.flowControlWindow = flowControlWindow;
            return this;
        }

        public GrpcOptions build() {
            return new GrpcOptions(this);
        }
    }
}
