package ru.yandex.kikimr.grpc;

import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Nullable;

import com.google.common.net.HostAndPort;
import com.google.common.util.concurrent.MoreExecutors;
import io.grpc.CallOptions;
import io.grpc.ClientCall;
import io.grpc.ConnectivityState;
import io.grpc.ManagedChannel;
import io.grpc.Metadata;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import io.grpc.netty.NegotiationType;
import io.grpc.netty.NettyChannelBuilder;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollSocketChannel;
import io.netty.channel.kqueue.KQueueEventLoopGroup;
import io.netty.channel.kqueue.KQueueSocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.kikimr.client.KikimrAsyncRetry;
import ru.yandex.kikimr.client.KikimrAsyncRetry.RetryConfig;
import ru.yandex.kikimr.client.UnableToConnectException;
import ru.yandex.kikimr.client.discovery.Discovery;
import ru.yandex.misc.actor.ActorRunner;

/**
 * TODO: add options class with fluent interface
 *
 * @author Sergey Polovko
 */
public final class GrpcTransport implements AutoCloseable {
    private static final RetryConfig RETRY_CONFIG = RetryConfig.maxRetries(5);
    private static final Logger logger = LoggerFactory.getLogger(GrpcTransport.class);

    private final GrpcOptions opts;
    private final Discovery discovery;
    private final ActorRunner actor;

    private CompletionStage<Void> onChange;
    private volatile ManagedChannel[] channels = new ManagedChannel[0];
    private final ConcurrentMap<HostAndPort, ManagedChannel> channelByNode = new ConcurrentHashMap<>();
    private final AtomicInteger next = new AtomicInteger();

    public GrpcTransport(Discovery discovery, GrpcOptions opts) {
        this.discovery = discovery;
        this.opts = opts;
        this.actor = new ActorRunner(this::act, MoreExecutors.directExecutor());
        actor.schedule();
    }

    private void act() {
        var actualOnChange = discovery.onChange();
        if (onChange != actualOnChange) {
            onChange = actualOnChange;
            onChange.whenComplete((ignore, e) -> actor.schedule());
        }

        var addresses = discovery.addresses();
        boolean hasAdded = addNewNodes(addresses);
        boolean hasRemoved = removeOldNodes(addresses);

        if (hasAdded || hasRemoved) {
            channels = channelByNode.values().toArray(new ManagedChannel[0]);
        }
    }

    private boolean addNewNodes(Set<HostAndPort> addresses) {
        boolean dirty = false;
        for (var address : addresses) {
            if (channelByNode.containsKey(address)) {
                continue;
            }

            try {
                var channel = createChannel(address, opts);
                channelByNode.put(address, channel);
                dirty = true;
            } catch (Throwable e) {
                logger.error("Unable to create channel to address {}", address, e);
            }
        }
        return dirty;
    }

    private boolean removeOldNodes(Set<HostAndPort> addresses) {
        boolean dirty = false;
        var it = channelByNode.entrySet().iterator();
        while (it.hasNext()) {
            var entry = it.next();
            var address = entry.getKey();
            if (addresses.contains(address)) {
                continue;
            }

            var node = entry.getValue();
            it.remove();
            try {
                node.shutdown();
            } catch (Throwable e) {
                logger.error("Unable to shutdown channel to address {}", address, e);
            }
            dirty = true;
        }
        return dirty;
    }

    private static ManagedChannel createChannel(HostAndPort address, GrpcOptions opts) {
        NettyChannelBuilder b = NettyChannelBuilder.forAddress(address.getHost(), address.getPort())
            .negotiationType(NegotiationType.PLAINTEXT)
            .maxInboundMessageSize(opts.maxMessageSizeBytes)
            .keepAliveTime(opts.keepAliveNanos, TimeUnit.NANOSECONDS)
            .keepAliveTimeout(opts.keepAliveTimeoutNanos, TimeUnit.NANOSECONDS)
            .idleTimeout(opts.idleTimeoutMillis, TimeUnit.MILLISECONDS);
        opts.channelOptions.forEach(b::withOption);

        if (opts.flowControlWindow != 0) {
            b.flowControlWindow(opts.flowControlWindow);
        }

        if (opts.ioExecutor != null) {
            b.eventLoopGroup(opts.ioExecutor);

            if (opts.ioExecutor instanceof EpollEventLoopGroup) {
                b.channelType(EpollSocketChannel.class);
            } else if (opts.ioExecutor instanceof KQueueEventLoopGroup) {
                b.channelType(KQueueSocketChannel.class);
            } else {
                b.channelType(NioSocketChannel.class);
            }
        }

        if (opts.callExecutor != null) {
            b.executor(opts.callExecutor);
        }
        opts.channelInitializer.accept(b);
        return b.build();
    }

    @Nullable
    private ManagedChannel nextReadyChannelSync() {
        // check every channel at most 2 times
        var channels = this.channels;
        for (int n = 0, count = channels.length; n < 2 * count; n++) {
            int index = Math.abs(next.incrementAndGet() % count);
            ManagedChannel channel = channels[index];
            if (channel.getState(true) == ConnectivityState.READY) {
                return channel;
            }
        }

        // no ready channel was found
        return null;
    }

    private CompletableFuture<ManagedChannel> nextReadyChannelAsync() {
        return KikimrAsyncRetry.withRetriesSync(RETRY_CONFIG, () -> {
            ManagedChannel channel = nextReadyChannelSync();
            if (channel == null) {
                throw noReadyChannelException(null);
            }
            return channel;
        });
    }

    public <ReqT, RespT> CompletableFuture<RespT> unaryCall(MethodDescriptor<ReqT, RespT> method, ReqT request, long expiredAt, @Nullable HostAndPort node) {
        final CallOptions callOptions;
        if (expiredAt != 0) {
            long remainingMillis = expiredAt - System.currentTimeMillis();
            if (remainingMillis <= 0) {
                return CompletableFuture.failedFuture(Status.DEADLINE_EXCEEDED.asRuntimeException());
            }
            callOptions = CallOptions.DEFAULT.withDeadlineAfter(remainingMillis, TimeUnit.MILLISECONDS);
        } else if (opts.readTimeoutMillis > 0) {
            callOptions = CallOptions.DEFAULT.withDeadlineAfter(opts.readTimeoutMillis, TimeUnit.MILLISECONDS);
        } else {
            callOptions = CallOptions.DEFAULT;
        }

        // (1) try to connect to a specific node
        if (node != null) {
            ManagedChannel channel = channelByNode.get(node);
            if (channel != null) {
                return callWithPromise(channel.newCall(method, callOptions), request);
            }
            return CompletableFuture.failedFuture(noReadyChannelException(node));
        }

        // (2) fast pass to get ready channel synchronously
        ManagedChannel channel = nextReadyChannelSync();
        if (channel != null) {
            return callWithPromise(channel.newCall(method, callOptions), request);
        }

        // (3) await ready channel with retires
        return nextReadyChannelAsync()
            .thenCompose(ch -> callWithPromise(ch.newCall(method, callOptions), request));
    }

    private UnableToConnectException noReadyChannelException(@Nullable HostAndPort node) {
        return new UnableToConnectException("No ready channel to communicate with " + node);
    }

    public Discovery discovery() {
        return discovery;
    }

    private static <ReqT, RespT> CompletableFuture<RespT> callWithPromise(ClientCall<ReqT, RespT> call, ReqT request) {
        CompletableFuture<RespT> promise = new CompletableFuture<>();
        try {
            call.start(new UnaryStreamToFuture<>(promise), new Metadata());
            call.request(1);
            call.sendMessage(request);
            call.halfClose();
        } catch (Throwable t) {
            call.cancel(null, t);
            promise.completeExceptionally(t);
        }
        return promise;
    }

    @Override
    public void close() {
        for (ManagedChannel channel : channels) {
            channel.shutdown();
        }
    }
}
